/**
 * Schema.org Structured Data Utilities
 * Provides helper functions for generating JSON-LD structured data
 */

export interface SchemaConfig {
  domain: string;
  siteName: string;
  description: string;
  logoUrl: string;
}

export const schemaConfig: SchemaConfig = {
  domain: process.env.NEXT_PUBLIC_DOMAIN || "https://your-domain.com",
  siteName: "برنامج الدبلوماسية الشبابية",
  description: "استمارة التقدم لبرنامج الدبلوماسية الشبابية",
  logoUrl: `${
    process.env.NEXT_PUBLIC_DOMAIN || "https://your-domain.com"
  }/android-icon-192x192.png`,
};

/**
 * Generate Organization schema
 */
export function generateOrganizationSchema() {
  return {
    "@context": "https://schema.org",
    "@type": "Organization",
    name: schemaConfig.siteName,
    url: schemaConfig.domain,
    logo: schemaConfig.logoUrl,
    description: schemaConfig.description,
    sameAs: [
      "https://www.facebook.com/YouthDiplomacy",
      "https://twitter.com/YouthDiplomacy",
    ],
    contactPoint: {
      "@type": "ContactPoint",
      contactType: "Customer Support",
      availableLanguage: ["ar", "en"],
    },
  };
}

/**
 * Generate BreadcrumbList schema
 */
export function generateBreadcrumbSchema(
  items: Array<{ name: string; url: string }>
) {
  return {
    "@context": "https://schema.org",
    "@type": "BreadcrumbList",
    itemListElement: items.map((item, index) => ({
      "@type": "ListItem",
      position: index + 1,
      name: item.name,
      item: item.url,
    })),
  };
}

/**
 * Generate WebPage schema
 */
export function generateWebPageSchema(
  title: string,
  description: string,
  url: string,
  imageUrl?: string
) {
  return {
    "@context": "https://schema.org",
    "@type": "WebPage",
    name: title,
    description: description,
    url: url,
    image: imageUrl || schemaConfig.logoUrl,
    mainEntity: {
      "@type": "Organization",
      name: schemaConfig.siteName,
      url: schemaConfig.domain,
    },
  };
}

/**
 * Generate FAQPage schema
 */
export function generateFAQSchema(
  faqs: Array<{ question: string; answer: string }>
) {
  return {
    "@context": "https://schema.org",
    "@type": "FAQPage",
    mainEntity: faqs.map((faq) => ({
      "@type": "Question",
      name: faq.question,
      acceptedAnswer: {
        "@type": "Answer",
        text: faq.answer,
      },
    })),
  };
}
