import { z } from "zod";
import { FormConfig } from "./types";

export function generateValidationSchema(config: FormConfig) {
  const schemaFields: any = {};

  config.sections.forEach((section) => {
    section.fields.forEach((field) => {
      let fieldSchema: any;

      // Handle nested documents object
      if (field.name === "documents" && field.objectFields) {
        const documentFields: any = {};

        field.objectFields.forEach((docField) => {
          if (docField.type === "file") {
            // Accept File objects or arrays of File objects
            documentFields[docField.name] = z
              .any()
              .refine(
                (files: any) => {
                  // Check if it's an array of Files
                  if (Array.isArray(files)) {
                    return (
                      files.length > 0 && files.every((f) => f instanceof File)
                    );
                  }
                  // Check if it's a single File
                  if (files instanceof File) {
                    return true;
                  }
                  return false;
                },
                { message: "File is required" }
              )
              .optional();
          }
        });

        schemaFields[field.name] = z.object(documentFields).optional();
        return;
      }

      // Handle array fields (nationalities, degrees, languages)
      if (field.type === "array" && field.arrayFields) {
        const arrayItemSchema: any = {};

        field.arrayFields.forEach((arrayField) => {
          if (arrayField.required) {
            if (arrayField.type === "select") {
              arrayItemSchema[arrayField.name] = z.string().min(1, {
                message: arrayField.errorMessage || "This field is required",
              });
            } else {
              arrayItemSchema[arrayField.name] = z.string().min(1, {
                message: arrayField.errorMessage || "This field is required",
              });
            }
          } else {
            arrayItemSchema[arrayField.name] = z.string().optional();
          }
        });

        fieldSchema = field.required
          ? z.array(z.object(arrayItemSchema)).min(1, {
              message:
                field.errorMessages?.required ||
                "At least one item is required",
            })
          : z.array(z.object(arrayItemSchema)).optional();

        schemaFields[field.name] = fieldSchema;
        return;
      }

      // Handle regular fields
      switch (field.type) {
        case "email":
          fieldSchema = z.string().email({
            message: field.errorMessages?.email || "Invalid email address",
          });
          if (field.required) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessages?.required || "Email is required",
            });
          }
          break;

        case "password":
          fieldSchema = z.string();
          if (field.required) {
            const minLength = field.validation?.min || 8;
            fieldSchema = fieldSchema.min(minLength, {
              message:
                field.errorMessages?.min ||
                `Password must be at least ${minLength} characters`,
            });
          }
          break;

        case "date":
          fieldSchema = z.string();
          if (field.required) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessages?.required || "Date is required",
            });
          }
          break;

        case "number":
          fieldSchema = z.coerce
            .number()
            .refine((val) => typeof val === "number" && !isNaN(val), {
              message: field.errorMessages?.numeric || "Must be a number",
            });
          if (field.required) {
            fieldSchema = fieldSchema.refine((val: number) => val > 0, {
              message: field.errorMessages?.required || "Number is required",
            });
          }
          break;

        case "url":
          fieldSchema = z.string().url({
            message: field.errorMessages?.url || "Invalid URL",
          });
          if (field.required) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessages?.required || "URL is required",
            });
          }
          break;

        case "file": {
          const fileRefinement = (files: any) => {
            if (typeof FileList !== "undefined" && files instanceof FileList) {
              return files.length > 0;
            }
            // Accept empty or undefined in SSR, or check for array-like
            return files && files.length > 0;
          };
          if (field.multiple) {
            fieldSchema = field.required
              ? z.any().refine(fileRefinement, {
                  message: field.errorMessages?.required || "Files required",
                })
              : z.any().optional();
          } else {
            fieldSchema = field.required
              ? z.any().refine(fileRefinement, {
                  message: field.errorMessages?.required || "File required",
                })
              : z.any().optional();
          }
          break;
        }

        case "radio":
          if (Array.isArray(field.options) && field.options.length > 0) {
            const firstOption = field.options[0];
            if (typeof firstOption === "object" && "value" in firstOption) {
              // Handle boolean options
              fieldSchema = z.union([z.boolean(), z.string()]);
            } else {
              fieldSchema = z.string();
            }
          } else {
            fieldSchema = z.string();
          }
          if (field.required) {
            fieldSchema = z.string().min(1, {
              message: field.errorMessage || "This field is required",
            });
          }
          break;

        case "select":
          fieldSchema = z.string();
          if (field.required && !field.conditional) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessage || "This field is required",
            });
          }
          // Validate against allowed options if specified
          if (
            field.validation?.in &&
            Array.isArray(field.validation.in) &&
            field.validation.in.length > 0
          ) {
            // z.enum requires a tuple type, so cast accordingly
            fieldSchema = z.enum(field.validation.in as [string, ...string[]], {
              message: field.errorMessages?.in || "Invalid selection",
            });
          }
          break;

        case "textarea":
          fieldSchema = z.string();
          if (field.required) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessage || "This field is required",
            });
          }
          break;

        default:
          fieldSchema = z.string();
          if (field.required && !field.conditional) {
            fieldSchema = fieldSchema.min(1, {
              message: field.errorMessage || "This field is required",
            });
          }

          // Handle pattern validation (like national_id digits)
          if (field.pattern) {
            fieldSchema = fieldSchema.regex(new RegExp(field.pattern), {
              message:
                field.errorMessages?.digits ||
                `Invalid format for ${field.label}`,
            });
          }

          // Handle specific validations
          if (field.validation?.digits) {
            fieldSchema = fieldSchema.length(field.validation.digits, {
              message:
                field.errorMessages?.digits ||
                `Must be ${field.validation.digits} digits`,
            });
          }
      }

      // Handle conditional fields - make them optional
      if (field.conditional) {
        schemaFields[field.name] = fieldSchema.optional();
      } else if (field.required) {
        schemaFields[field.name] = fieldSchema;
      } else {
        schemaFields[field.name] = fieldSchema.optional();
      }
    });
  });

  return z.object(schemaFields);
}
