import axios, { AxiosInstance, AxiosError } from "axios";

const API_BASE_URL =
  process.env.NEXT_PUBLIC_API_URL || "http://localhost:8000/api/v1";

// Create axios instance
const apiClient: AxiosInstance = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    "Content-Type": "application/json",
    Accept: "application/json",
  },
});

// Request interceptor to add auth token
apiClient.interceptors.request.use(
  (config) => {
    const token = localStorage.getItem("auth_token");
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }
    return config;
  },
  (error) => Promise.reject(error)
);

// Response interceptor for error handling
apiClient.interceptors.response.use(
  (response) => response,
  (error: AxiosError) => {
    if (error.response?.status === 401) {
      // Clear token and redirect to login
      localStorage.removeItem("auth_token");
      localStorage.removeItem("user_data");
      if (typeof window !== "undefined") {
        window.location.href = "/login";
      }
    }
    return Promise.reject(error);
  }
);

// Auth APIs
export const authAPI = {
  login: (email: string, password: string) =>
    apiClient.post("/login", { email, password }),

  logout: () => apiClient.post("/logout"),

  getProfile: () => apiClient.get("/me"),

  getApplicantData: () => apiClient.get("/applicant"),

  downloadDocument: (documentId: number) =>
    apiClient.get(`/documents/${documentId}/download`, {
      responseType: "blob",
    }),

  changePassword: (data: {
    current_password: string;
    new_password: string;
    new_password_confirmation: string;
  }) => apiClient.post("/change-password", data),
};

// User APIs
export const userAPI = {
  getExams: () => apiClient.get("/exams"),

  getExam: (examId: number) => apiClient.get(`/exams/${examId}`),

  getExamResults: (examId: number) => apiClient.get(`/exams/${examId}/results`),

  startExam: (examId: number) => apiClient.post(`/exams/${examId}/start`),

  submitExam: (
    examId: number,
    answers: Record<string, string>,
    autoSubmit: boolean = false
  ) =>
    apiClient.post(`/exams/${examId}/submit`, {
      answers,
      auto_submit: autoSubmit,
    }),
};

// Judge APIs
export const judgeAPI = {
  // Evaluation Sheets (judges can only view active sheets)
  getEvaluationSheets: () => apiClient.get("/judge/evaluation-sheets"),

  // Users Evaluation
  getAssignedUsers: (filters?: {
    page?: number;
    per_page?: number;
    search?: string;
  }) => apiClient.get("/judge/users", { params: filters }),

  evaluateUser: (
    userId: number,
    data: {
      evaluation_sheet_id: number;
      scores: Record<string, number>;
      notes?: string;
    }
  ) => apiClient.post(`/judge/users/${userId}/evaluate`, data),

  getMyEvaluations: (filters?: any) =>
    apiClient.get("/judge/evaluations", { params: filters }),

  // User Details
  getUserDetail: (id: number) => apiClient.get(`/judge/users/${id}`),

  getUserApplicantData: (id: number) =>
    apiClient.get(`/judge/users/${id}/applicant-data`),

  getUserExamResults: (id: number) =>
    apiClient.get(`/judge/users/${id}/exam-results`),

  // Statistics
  getStatistics: () => apiClient.get("/judge/statistics"),
};

// Admin APIs
export const adminAPI = {
  // Evaluation Sheet Management (admin only)
  getEvaluationSheets: () => apiClient.get("/admin/evaluation-sheets"),

  createEvaluationSheet: (data: {
    title: string;
    criteria: Array<{ name: string; max_score: number }>;
    is_active?: boolean;
  }) => apiClient.post("/admin/evaluation-sheets", data),

  getEvaluationSheet: (id: number) =>
    apiClient.get(`/admin/evaluation-sheets/${id}`),

  updateEvaluationSheet: (id: number, data: any) =>
    apiClient.put(`/admin/evaluation-sheets/${id}`, data),

  deleteEvaluationSheet: (id: number) =>
    apiClient.delete(`/admin/evaluation-sheets/${id}`),

  // Users Management
  getUsers: (filters?: {
    status?: string;
    category_id?: number;
    search?: string;
    page?: number;
  }) => apiClient.get("/admin/users", { params: filters }),

  getUser: (id: number) => apiClient.get(`/admin/users/${id}`),

  getUserExamResults: (id: number) =>
    apiClient.get(`/admin/users/${id}/exam-results`),

  getApplicantData: (id: number) =>
    apiClient.get(`/admin/users/${id}/applicant-data`),

  acceptUser: (id: number, data?: { notes?: string }) =>
    apiClient.post(`/admin/users/${id}/accept`, data),

  rejectUser: (
    id: number,
    data: { rejection_reason: string; notes?: string }
  ) => apiClient.post(`/admin/users/${id}/reject`, data),

  exportUsers: (filters?: {
    status?: string;
    category_id?: number;
    search?: string;
  }) =>
    apiClient.get("/admin/users/export/excel", {
      params: filters,
      responseType: "blob",
    }),

  // Categories
  getCategories: () => apiClient.get("/admin/categories"),

  createCategory: (data: { name: string; description?: string }) =>
    apiClient.post("/admin/categories", data),

  updateCategory: (id: number, data: { name?: string; description?: string }) =>
    apiClient.put(`/admin/categories/${id}`, data),

  deleteCategory: (id: number) => apiClient.delete(`/admin/categories/${id}`),

  assignUserToCategory: (categoryId: number, userId: number) =>
    apiClient.post(`/admin/categories/${categoryId}/users`, {
      user_id: userId,
    }),

  removeUserFromCategory: (categoryId: number, userId: number) =>
    apiClient.delete(`/admin/categories/${categoryId}/users/${userId}`),

  bulkAssignUsersToCategory: (categoryId: number, userIds: number[]) =>
    apiClient.post(`/admin/categories/${categoryId}/bulk-assign`, {
      user_ids: userIds,
    }),

  bulkRemoveUsersFromCategory: (categoryId: number, userIds: number[]) =>
    apiClient.post(`/admin/categories/${categoryId}/bulk-remove`, {
      user_ids: userIds,
    }),

  assignCategoriesToUser: (userId: number, categoryIds: number[]) =>
    apiClient.post(`/admin/users/${userId}/assign-categories`, {
      category_ids: categoryIds,
    }),

  removeCategoriesFromUser: (userId: number, categoryIds: number[]) =>
    apiClient.post(`/admin/users/${userId}/remove-categories`, {
      category_ids: categoryIds,
    }),

  // Exams
  createExam: (data: {
    title: string;
    description?: string;
    form_schema: any;
    deadline_at?: string | null;
  }) => apiClient.post("/admin/exams", data),

  getExams: () => apiClient.get("/admin/exams"),

  getExam: (id: number) => apiClient.get(`/admin/exams/${id}`),

  updateExam: (
    id: number,
    data: {
      title?: string;
      description?: string;
      form_schema?: any;
      deadline_at?: string | null;
    }
  ) => apiClient.put(`/admin/exams/${id}`, data),

  deleteExam: (id: number) => apiClient.delete(`/admin/exams/${id}`),

  publishExam: (id: number) => apiClient.post(`/admin/exams/${id}/publish`),

  unpublishExam: (id: number) => apiClient.post(`/admin/exams/${id}/unpublish`),

  // System Logs
  getLogs: (filters?: any) => apiClient.get("/admin/logs", { params: filters }),
  getLogActions: () => apiClient.get("/admin/logs/actions"),
  getLogStatistics: (filters?: any) =>
    apiClient.get("/admin/logs/statistics", { params: filters }),

  // Settings
  getSettings: () => apiClient.get("/admin/settings"),
  updateSettings: (settings: Record<string, any>) =>
    apiClient.put("/admin/settings", { settings }),
  getSetting: (key: string) => apiClient.get(`/admin/settings/${key}`),

  // Languages
  getLanguages: () => apiClient.get("/admin/languages"),

  createLanguage: (data: {
    name_en: string;
    name_ar: string;
    iso_code?: string;
  }) => apiClient.post("/admin/languages", data),

  getLanguage: (id: number) => apiClient.get(`/admin/languages/${id}`),

  updateLanguage: (
    id: number,
    data: { name_en?: string; name_ar?: string; iso_code?: string }
  ) => apiClient.put(`/admin/languages/${id}`, data),

  deleteLanguage: (id: number) => apiClient.delete(`/admin/languages/${id}`),

  // Countries
  getCountries: () => apiClient.get("/admin/countries"),

  createCountry: (data: {
    name_en: string;
    name_ar: string;
    iso_code?: string;
  }) => apiClient.post("/admin/countries", data),

  getCountry: (id: number) => apiClient.get(`/admin/countries/${id}`),

  updateCountry: (
    id: number,
    data: { name_en?: string; name_ar?: string; iso_code?: string }
  ) => apiClient.put(`/admin/countries/${id}`, data),

  deleteCountry: (id: number) => apiClient.delete(`/admin/countries/${id}`),

  // Governorates
  getGovernorates: () => apiClient.get("/admin/governorates"),

  createGovernorate: (data: {
    name_en: string;
    name_ar: string;
    code?: string;
  }) => apiClient.post("/admin/governorates", data),

  getGovernorate: (id: number) => apiClient.get(`/admin/governorates/${id}`),

  updateGovernorate: (
    id: number,
    data: { name_en?: string; name_ar?: string; code?: string }
  ) => apiClient.put(`/admin/governorates/${id}`, data),

  deleteGovernorate: (id: number) =>
    apiClient.delete(`/admin/governorates/${id}`),

  // Statistics
  getStatistics: () => apiClient.get("/admin/statistics"),
};

// Helper function to handle API errors
export const handleAPIError = (error: any): string => {
  if (error.response?.data?.message) {
    return error.response.data.message;
  }
  if (error.response?.data?.errors) {
    const errors = error.response.data.errors;
    return Object.values(errors).flat().join(", ");
  }
  return error.message || "حدث خطأ غير متوقع";
};

export default apiClient;
