"use client";

import React, { useState, useEffect } from "react";
import { FormRenderer } from "@/components/form/FormRenderer";
import { FormConfig } from "@/lib/types";
import {
  generateBrowserFingerprint,
  getDeviceType,
  getOS,
  getBrowser,
} from "@/lib/fingerprint";
import axios from "axios";

interface RegistrationGuardProps {
  config: FormConfig;
}

export function RegistrationGuard({ config }: RegistrationGuardProps) {
  const [status, setStatus] = useState<
    "checking" | "allowed" | "already_applied"
  >("checking");
  const [applicantCode, setApplicantCode] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState<string | null>(null);

  useEffect(() => {
    // Auto-check on mount
    checkRegistrationStatus();
  }, []);

  const checkRegistrationStatus = async () => {
    // First check localStorage
    const storedCode = localStorage.getItem("applicant_code");
    const storedEmail = localStorage.getItem("applicant_email");

    if (storedCode) {
      setApplicantCode(storedCode);
      setUserEmail(storedEmail);
      setStatus("already_applied");
      return;
    }

    // Then check with server (IP/fingerprint based)
    try {
      const fingerprint = await generateBrowserFingerprint();

      const response = await axios.post(
        `${
          process.env.NEXT_PUBLIC_API_URL || "http://localhost:8000/api/v1"
        }/check-device-submission`,
        {
          browser_fingerprint: fingerprint,
        }
      );

      if (response.data.has_submitted && response.data.applicant_code) {
        setApplicantCode(response.data.applicant_code);
        // Store in localStorage for next time
        localStorage.setItem("applicant_code", response.data.applicant_code);
        if (response.data.email) {
          setUserEmail(response.data.email);
          localStorage.setItem("applicant_email", response.data.email);
        }
        setStatus("already_applied");
        return;
      }
    } catch (error) {
      console.error("Error checking device submission:", error);
      // On error, continue to allow form (fail open)
    }

    // If no stored code and no server match, allow them to proceed to form
    setStatus("allowed");
  };

  const handleAlreadyApplied = (code: string, email?: string) => {
    // Store the registration info after successful submission
    localStorage.setItem("applicant_code", code);
    if (email) {
      localStorage.setItem("applicant_email", email);
      setUserEmail(email);
    }
    setApplicantCode(code);
    setStatus("already_applied");
  };

  if (status === "checking") {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="text-gray-600 mt-4">جاري التحقق من البيانات...</p>
        </div>
      </div>
    );
  }

  if (status === "already_applied") {
    return (
      <div
        className="max-w-2xl mx-auto text-center bg-white rounded-lg shadow-xl p-12 mb-12"
        dir="rtl"
      >
        <div className="mb-8">
          <svg
            className="w-24 h-24 mx-auto text-amber-400"
            fill="none"
            stroke="currentColor"
            viewBox="0 0 24 24"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M12 8v4m0 4v.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
            />
          </svg>
        </div>
        <h1 className="text-4xl font-bold text-gray-900 mb-4">
          لقد قمت بالتقديم من قبل!
        </h1>
        <p className="text-lg text-gray-700 mb-6">
          شكراً لك على اهتمامك ببرنامج الدبلوماسية الشبابية.
        </p>
        <div className="bg-amber-50 border-2 border-amber-200 rounded-lg p-6 mb-6">
          <p className="text-gray-700 mb-3">رمز طلبك:</p>
          <p className="text-3xl font-bold text-amber-600 mb-3">
            {applicantCode}
          </p>
          <p className="text-sm text-gray-600">
            احتفظ بهذا الرمز للاستعلام عن حالة طلبك
          </p>
          {userEmail && (
            <p className="text-sm text-gray-600 mt-2">
              البريد الإلكتروني: {userEmail}
            </p>
          )}
        </div>
        <p className="text-gray-600 mb-8">
          لا يمكنك التقديم أكثر من مرة واحدة في نفس الدورة، طلبك قد تم حفظه
          والرجاء الانتظار لتقييمه من قبل فريقنا.
        </p>
      </div>
    );
  }

  return (
    <FormRenderer config={config} onAlreadyApplied={handleAlreadyApplied} />
  );
}
