"use client";

import React, { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { FormConfig } from "@/lib/types";
import { generateValidationSchema } from "@/lib/formSchema";
import { generateApplicantCode, convertToBase64, apiCall } from "@/lib/utils";
import {
  generateBrowserFingerprint,
  getDeviceType,
  getOS,
  getBrowser,
} from "@/lib/fingerprint";
import { FormField } from "./FormField";
import { ProgressBar } from "./ProgressBar";
import { Button } from "@/components/ui/Button";
import { ChevronLeft, ChevronRight, Send, CheckCircle } from "lucide-react";
import axios from "axios";
import {
  Download,
  Mail,
  Phone,
  MapPin,
  Calendar,
  User,
  GraduationCap,
  Briefcase,
  Globe,
  Award,
  FileText,
  Users,
  Languages,
} from "lucide-react";

interface FormRendererProps {
  config: FormConfig;
  onAlreadyApplied?: (applicantCode: string, email?: string) => void;
}

export function FormRenderer({ config, onAlreadyApplied }: FormRendererProps) {
  const [currentSection, setCurrentSection] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitSuccess, setSubmitSuccess] = useState(false);
  const [submitError, setSubmitError] = useState("");
  const [applicantCode] = useState(generateApplicantCode());
  const [data, setSuccessData] = useState<any>(null);

  const schema = generateValidationSchema(config);
  const {
    register,
    handleSubmit,
    watch,
    setValue,
    trigger,
    getValues,
    control,
    formState: { errors, isValidating },
  } = useForm({
    resolver: zodResolver(schema),
    mode: "onChange",
    reValidateMode: "onChange",
    defaultValues: {
      nationalities: [],
      degrees: [],
      languages: [],
      documents: {
        national_id: null,
        cv: null,
        photo_casual: null,
        photo_official: null,
      },
    },
  });

  // Watch all form values to trigger re-renders when values change
  const allFormValues = watch();

  const onSubmit = async (data: any) => {
    setIsSubmitting(true);
    setSubmitError("");

    try {
      // Generate browser fingerprint
      const browserFingerprint = await generateBrowserFingerprint();
      const deviceType = getDeviceType();
      const os = getOS();
      const browser = getBrowser();

      // Create FormData for multipart/form-data submission
      const formData = new FormData();

      // Process data and add to FormData
      for (const key in data) {
        const value = data[key];

        // Handle documents object with files
        if (
          key === "documents" &&
          typeof value === "object" &&
          value !== null
        ) {
          for (const docKey in value) {
            const files = value[docKey];

            // Skip if no files
            if (!files) {
              continue;
            }

            // photo_casual is the only multi-file field
            if (docKey === "photo_casual" && Array.isArray(files)) {
              const validFiles = files.filter((f) => f instanceof File);
              validFiles.forEach((file) => {
                formData.append(`documents[${docKey}][]`, file);
              });
            }
            // All other document fields are single files
            else if (Array.isArray(files)) {
              const validFiles = files.filter((f) => f instanceof File);
              if (validFiles.length > 0) {
                // Take the first file only for single-file fields
                formData.append(`documents[${docKey}]`, validFiles[0]);
              }
            }
            // Handle single File
            else if (files instanceof File) {
              formData.append(`documents[${docKey}]`, files);
            }
          }
        }
        // Handle arrays (nationalities, languages, degrees)
        else if (Array.isArray(value)) {
          value.forEach((item, index) => {
            if (typeof item === "object" && item !== null) {
              for (const itemKey in item) {
                formData.append(
                  `${key}[${index}][${itemKey}]`,
                  String(item[itemKey])
                );
              }
            } else {
              formData.append(`${key}[${index}]`, String(item));
            }
          });
        }
        // Handle regular fields
        else if (value !== null && value !== undefined) {
          // Handle boolean fields (is_studying, grad_year - stored as strings "true"/"false")
          if (key === "is_studying") {
            // Convert "true"/"false" string to 1/0 for Laravel boolean
            formData.append(
              key,
              value === "true" || value === true ? "1" : "0"
            );
          }
          // Convert other booleans to string for FormData
          else if (typeof value === "boolean") {
            formData.append(key, value ? "1" : "0");
          } else {
            formData.append(key, String(value));
          }
        }
      }

      // Add applicant code and device fingerprint
      formData.append("applicant_code", applicantCode);
      formData.append("browser_fingerprint", browserFingerprint);
      formData.append("device_type", deviceType);
      formData.append("os", os);
      formData.append("browser", browser);

      // Submit to API using axios with base URL
      const apiBaseUrl =
        process.env.NEXT_PUBLIC_API_URL || "http://localhost:8000/api/v1";
      const response = await axios.post(`${apiBaseUrl}/applicants`, formData, {
        headers: {
          "Content-Type": "multipart/form-data",
        },
      });

      if (response.status === 200 || response.status === 201) {
        // Extract applicant data from nested response structure
        const applicantData = response.data.data?.applicant || response.data;
        const applicantCodeResponse = applicantData.applicant_code;
        const email = applicantData.email || data.email;

        // Store in localStorage immediately to prevent re-submission on refresh
        localStorage.setItem("applicant_code", applicantCodeResponse);
        localStorage.setItem("applicant_email", email);

        // DON'T call callback on success - let FormRenderer show its own success page
        // The callback is only for when user tries to resubmit (409 error)

        setSuccessData(applicantData);
        setSubmitSuccess(true);
        window.scrollTo({ top: 0, behavior: "smooth" });
      }
    } catch (error: any) {
      console.error("Submission error:", error);
      console.error("Response data:", error.response?.data);
      console.error("Response status:", error.response?.status);

      // Check if maintenance mode
      if (
        error.response?.data?.maintenance_mode ||
        error.response?.status === 503
      ) {
        setSubmitError("النظام في وضع الصيانة حالياً");
        setTimeout(() => {
          window.location.href = "/maintenance";
        }, 2000);
        return;
      }

      // Check if already applied
      if (
        error.response?.status === 409 &&
        error.response?.data?.error_code === "ALREADY_APPLIED"
      ) {
        setSubmitError(error.response.data.message);
        // Call the callback if provided to notify parent component
        if (onAlreadyApplied) {
          onAlreadyApplied(error.response.data.applicant_code, data.email);
        }
        window.scrollTo({ top: 0, behavior: "smooth" });
        return;
      }

      // Check if registration is closed
      if (error.response?.data?.registration_closed) {
        setSubmitError("لقد تم انتهاء مدة التسجيل");
        // Reload page after 2 seconds to show closed message
        setTimeout(() => {
          window.location.reload();
        }, 2000);
        return;
      }

      // Handle Laravel validation errors
      if (error.response?.data?.errors) {
        const errorMessages = Object.values(error.response.data.errors)
          .flat()
          .join(" ");
        setSubmitError(errorMessages);
      } else {
        setSubmitError(
          error.response?.data?.message ||
            "Submission failed. Please try again."
        );
      }

      window.scrollTo({ top: 0, behavior: "smooth" });
    } finally {
      setIsSubmitting(false);
    }
  };

  const nextSection = async () => {
    if (currentSection < config.sections.length - 1) {
      const currentSectionFields = config.sections[currentSection].fields;

      // Get field names to validate
      const fieldNames = currentSectionFields
        .filter((field) => {
          // Only validate fields that should be shown (non-conditional or condition met)
          if (field.conditional) {
            const watchValue = watch(field.conditional.field);
            const normalizedValue =
              watchValue === "true"
                ? true
                : watchValue === "false"
                ? false
                : watchValue;
            return typeof field.conditional.value === "boolean"
              ? normalizedValue === field.conditional.value
              : String(watchValue) === String(field.conditional.value);
          }
          return true;
        })
        .map((field) => field.name);

      // Trigger validation for current section fields
      // Add a small delay to ensure file inputs are properly registered
      await new Promise((resolve) => setTimeout(resolve, 100));
      const isValid = await trigger(fieldNames);

      if (!isValid) {
        // Show error message
        setSubmitError("يرجى ملء جميع الحقول المطلوبة في هذا القسم بشكل صحيح");
        window.scrollTo({ top: 0, behavior: "smooth" });
        return;
      }

      setSubmitError("");
      setCurrentSection(currentSection + 1);
      window.scrollTo({ top: 0, behavior: "smooth" });
    }
  };

  const prevSection = () => {
    if (currentSection > 0) {
      setCurrentSection(currentSection - 1);
      window.scrollTo({ top: 0, behavior: "smooth" });
    }
  };

  // Helper function to check if current section has validation errors
  // Check if a specific section is valid
  const isSectionValidByIndex = (sectionIndex: number) => {
    const sectionFields = config.sections[sectionIndex].fields;
    const formValues = getValues();

    // Check if any required field in this section has an error
    for (const field of sectionFields) {
      // Skip conditional fields if condition is not met
      if (field.conditional) {
        const watchValue = watch(field.conditional.field);
        const normalizedValue =
          watchValue === "true"
            ? true
            : watchValue === "false"
            ? false
            : watchValue;
        const conditionMet =
          typeof field.conditional.value === "boolean"
            ? normalizedValue === field.conditional.value
            : String(watchValue) === String(field.conditional.value);

        if (!conditionMet) continue;
      }

      if (field.required) {
        const fieldValue = (formValues as any)[field.name];

        // Handle documents
        if (field.name === "documents" && field.objectFields) {
          if (!fieldValue || typeof fieldValue !== "object") return false;
          for (const docField of field.objectFields) {
            if (docField.required) {
              const docValue = (fieldValue as any)[docField.name];
              if (!docValue) return false;
              if (Array.isArray(docValue) && docValue.length === 0)
                return false;
            }
          }
        }
        // Handle array fields
        else if (field.type === "array" && Array.isArray(fieldValue)) {
          if (fieldValue.length === 0) return false;
        }
        // Handle regular fields
        else if (!fieldValue || fieldValue === "") {
          return false;
        }
      }
    }

    return true;
  };

  // Check current section validity
  const isSectionValid = () => isSectionValidByIndex(currentSection);

  const InfoCard = ({
    title,
    children,
    icon: Icon,
  }: {
    title: string;
    children: React.ReactNode;
    icon: React.ComponentType<{ className?: string }>;
  }) => (
    <div className="bg-white rounded-xl shadow-lg p-6 mb-6">
      <div className="flex items-center gap-3 mb-6 border-b pb-4">
        {Icon && <Icon className="w-6 h-6 text-blue-600" />}
        <h3 className="text-2xl font-bold text-gray-800">{title}</h3>
      </div>
      {children}
    </div>
  );

  const InfoItem = ({
    label,
    value,
    fullWidth = false,
  }: {
    label: string;
    value?: string | number;
    fullWidth?: boolean;
  }) => (
    <div className={fullWidth ? "md:col-span-2" : ""}>
      <p className="text-gray-600 text-sm mb-1">{label}</p>
      <p className="text-lg font-semibold text-gray-800">{value || "-"}</p>
    </div>
  );

  const getProficiencyLabel = (proficiency: string): string => {
    const labels: Record<string, string> = {
      excellent: "ممتاز",
      very_good: "جيد جداً",
      good: "جيد",
      fair: "متوسط",
    };
    return labels[proficiency] || proficiency;
  };

  const getDocumentTypeLabel = (type: string): string => {
    const labels: Record<string, string> = {
      national_id: "البطاقة الشخصية",
      passport: "جواز السفر",
      cv: "السيرة الذاتية",
      photo_casual: "صورة شخصية",
      photo_official: "صورة رسمية",
      certificate: "الشهادة",
    };
    return labels[type] || type;
  };

  if (submitSuccess && data) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-green-50 py-8 px-4">
        <div className="max-w-6xl mx-auto">
          {/* Success Header */}
          <div className="text-center py-12 bg-gradient-to-r from-green-50 to-blue-50 rounded-2xl mb-8 shadow-lg">
            <CheckCircle className="w-24 h-24 text-green-500 mx-auto mb-6 animate-bounce" />
            <h1 className="text-4xl font-bold text-gray-800 mb-4">
              تم استقبال الطلب بنجاح!
            </h1>
            <p className="text-lg text-gray-600 mb-6 max-w-2xl mx-auto">
              شكراً لك على ملء نموذج الترشح. تم حفظ بيانات الطلب بنجاح وسيتم
              مراجعتها من قبل الفريق المختص.
            </p>

            {/* Application Code */}
            <div className="bg-white px-8 py-6 rounded-xl inline-block shadow-xl border-2 border-blue-200">
              <p className="text-gray-600 mb-2 text-sm">رمز الطلب الخاص بك</p>
              <p className="text-4xl font-bold text-blue-600 tracking-wider">
                {data.applicant_code}
              </p>
              <p className="text-sm text-gray-500 mt-3">
                احتفظ بهذا الرمز للمتابعة والاستعلام عن طلبك
              </p>
            </div>

            {/* Email Notification */}
            {/* <div className="mt-6 bg-blue-100 border border-blue-300 rounded-lg p-4 max-w-md mx-auto">
              <Mail className="w-5 h-5 text-blue-600 inline-block ml-2" />
              <span className="text-blue-800 text-sm">
                تم إرسال بيانات الدخول إلى: <strong>{data.email}</strong>
              </span>
            </div> */}
          </div>

          {/* Personal Information */}
          <InfoCard title="البيانات الشخصية" icon={User}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <InfoItem
                label="الاسم بالعربية"
                value={data.full_name_ar}
                fullWidth
              />
              <InfoItem
                label="الاسم بالإنجليزية"
                value={data.full_name_en}
                fullWidth
              />
              <InfoItem label="الرقم القومي" value={data.national_id} />
              {data.passport_number && (
                <InfoItem label="رقم جواز السفر" value={data.passport_number} />
              )}
              <InfoItem label="تاريخ الميلاد" value={data.dob} />
              <InfoItem label="مكان الميلاد" value={data.birth_place} />
              <InfoItem
                label="النوع"
                value={data.gender === "male" ? "ذكر" : "أنثى"}
              />
              {data.military_status && (
                <InfoItem
                  label="الحالة العسكرية"
                  value={data.military_status}
                />
              )}
              <InfoItem label="الطول (سم)" value={data.height} />
              <InfoItem label="الوزن (كجم)" value={data.weight} />
            </div>
          </InfoCard>

          {/* Contact Information */}
          <InfoCard title="معلومات الاتصال" icon={Phone}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <InfoItem label="رقم الهاتف" value={data.mobile} />
              {data.whatsapp && (
                <InfoItem label="واتساب" value={data.whatsapp} />
              )}
              <InfoItem
                label="البريد الإلكتروني"
                value={data.email}
                fullWidth
              />
              <InfoItem label="العنوان" value={data.address} fullWidth />
              <InfoItem label="الحي/المنطقة" value={data.district} />
              <InfoItem label="المحافظة" value={data.governorate?.name_ar} />
            </div>
          </InfoCard>

          {/* Nationalities */}
          {data.nationalities && data.nationalities.length > 0 && (
            <InfoCard title="الجنسيات" icon={Globe}>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                {data.nationalities.map((nat: any, idx: number) => (
                  <div
                    key={idx}
                    className="bg-gradient-to-br from-blue-50 to-indigo-50 p-4 rounded-lg border border-blue-200"
                  >
                    <p className="text-lg font-semibold text-gray-800">
                      {nat.country?.name_ar || nat.country?.name_en}
                    </p>
                    <p className="text-sm text-gray-600">
                      {nat.country?.iso_code}
                    </p>
                  </div>
                ))}
              </div>
            </InfoCard>
          )}

          {/* Education */}
          <InfoCard title="المؤهلات العلمية" icon={GraduationCap}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
              <InfoItem label="الجامعة" value={data.university} />
              <InfoItem label="الكلية" value={data.college} />
              <InfoItem label="التخصص" value={data.major} />
              <InfoItem label="الدرجة العلمية" value={data.degree} />
              {data.is_studying === "1" && (
                <InfoItem
                  label="السنة الدراسية الحالية"
                  value={data.study_year}
                  fullWidth
                />
              )}
            </div>

            {/* Additional Degrees */}
            {data.degrees && data.degrees.length > 0 && (
              <div className="pt-6 border-t">
                <h4 className="text-lg font-bold text-gray-800 mb-4 flex items-center gap-2">
                  <Award className="w-5 h-5 text-blue-600" />
                  الدرجات العلمية الإضافية
                </h4>
                <div className="space-y-3">
                  {data.degrees.map((deg: any, idx: number) => (
                    <div
                      key={idx}
                      className="bg-gradient-to-r from-gray-50 to-blue-50 p-4 rounded-lg border border-gray-200"
                    >
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                          <p className="text-xs text-gray-600">الدرجة</p>
                          <p className="font-semibold text-gray-800">
                            {deg.degree_title}
                          </p>
                        </div>
                        <div>
                          <p className="text-xs text-gray-600">المؤسسة</p>
                          <p className="font-semibold text-gray-800">
                            {deg.institution}
                          </p>
                        </div>
                        <div>
                          <p className="text-xs text-gray-600">السنة</p>
                          <p className="font-semibold text-gray-800">
                            {deg.year}
                          </p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </InfoCard>

          {/* Languages */}
          {data.languages && data.languages.length > 0 && (
            <InfoCard title="اللغات" icon={Languages}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {data.languages.map((lang: any, idx: number) => (
                  <div
                    key={idx}
                    className="bg-gradient-to-br from-purple-50 to-pink-50 p-5 rounded-lg border border-purple-200"
                  >
                    <p className="text-gray-600 text-sm mb-2">اللغة</p>
                    <p className="text-xl font-bold text-gray-800 mb-3">
                      {lang.language?.name_ar || lang.language?.name_en}
                    </p>
                    <div className="flex items-center gap-2">
                      <span className="text-sm text-gray-600">
                        مستوى الإجادة:
                      </span>
                      <span className="bg-purple-600 text-white px-3 py-1 rounded-full text-sm font-semibold">
                        {getProficiencyLabel(lang.proficiency)}
                      </span>
                    </div>
                  </div>
                ))}
              </div>
            </InfoCard>
          )}

          {/* Work Experience */}
          {(data.current_job ||
            data.employer ||
            data.experience_public ||
            data.experience_ministry ||
            data.experience_other_orgs) && (
            <InfoCard title="الخبرات العملية" icon={Briefcase}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {data.current_job && (
                  <InfoItem label="الوظيفة الحالية" value={data.current_job} />
                )}
                {data.employer && (
                  <InfoItem label="جهة العمل" value={data.employer} />
                )}
                {data.experience_public && (
                  <InfoItem
                    label="الخبرة في القطاع العام"
                    value={data.experience_public}
                    fullWidth
                  />
                )}
                {data.experience_ministry && (
                  <InfoItem
                    label="الخبرة في الوزارة"
                    value={data.experience_ministry}
                    fullWidth
                  />
                )}
                {data.experience_other_orgs && (
                  <InfoItem
                    label="الخبرة في مؤسسات أخرى"
                    value={data.experience_other_orgs}
                    fullWidth
                  />
                )}
              </div>
            </InfoCard>
          )}

          {/* Activities & Interests */}
          {(data.visited_countries ||
            data.center_memberships ||
            data.club_memberships ||
            data.sports ||
            data.skills_hobbies) && (
            <InfoCard title="الأنشطة والاهتمامات" icon={Users}>
              <div className="space-y-4">
                {data.visited_countries && (
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-2">
                      الدول التي تمت زيارتها
                    </p>
                    <p className="text-gray-800">{data.visited_countries}</p>
                  </div>
                )}
                {data.center_memberships && (
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-2">عضويات المراكز</p>
                    <p className="text-gray-800">{data.center_memberships}</p>
                  </div>
                )}
                {data.club_memberships && (
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-2">عضويات الأندية</p>
                    <p className="text-gray-800">{data.club_memberships}</p>
                  </div>
                )}
                {data.sports && (
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-2">الرياضات</p>
                    <p className="text-gray-800">{data.sports}</p>
                  </div>
                )}
                {data.skills_hobbies && (
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-sm text-gray-600 mb-2">
                      المهارات والهوايات
                    </p>
                    <p className="text-gray-800">{data.skills_hobbies}</p>
                  </div>
                )}
              </div>
            </InfoCard>
          )}

          {/* Personal Insights */}
          {(data.motivation || data.strengths || data.weaknesses) && (
            <InfoCard title="معلومات شخصية" icon={FileText}>
              <div className="space-y-4">
                {data.motivation && (
                  <div className="bg-blue-50 p-5 rounded-lg border border-blue-200">
                    <p className="text-sm text-blue-700 font-semibold mb-2">
                      الدافع للتقديم
                    </p>
                    <p className="text-gray-800 leading-relaxed">
                      {data.motivation}
                    </p>
                  </div>
                )}
                {data.strengths && (
                  <div className="bg-green-50 p-5 rounded-lg border border-green-200">
                    <p className="text-sm text-green-700 font-semibold mb-2">
                      نقاط القوة
                    </p>
                    <p className="text-gray-800 leading-relaxed">
                      {data.strengths}
                    </p>
                  </div>
                )}
                {data.weaknesses && (
                  <div className="bg-amber-50 p-5 rounded-lg border border-amber-200">
                    <p className="text-sm text-amber-700 font-semibold mb-2">
                      نقاط التحسين
                    </p>
                    <p className="text-gray-800 leading-relaxed">
                      {data.weaknesses}
                    </p>
                  </div>
                )}
              </div>
            </InfoCard>
          )}

          {/* Social Media */}
          {(data.facebook || data.instagram || data.linkedin) && (
            <InfoCard title="وسائل التواصل الاجتماعي" icon={Globe}>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                {data.facebook && (
                  <a
                    href={data.facebook}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="bg-blue-50 hover:bg-blue-100 p-4 rounded-lg border border-blue-200 transition-colors"
                  >
                    <p className="text-sm text-gray-600 mb-2">فيسبوك</p>
                    <p className="text-blue-600 font-semibold truncate">
                      الملف الشخصي
                    </p>
                  </a>
                )}
                {data.instagram && (
                  <a
                    href={data.instagram}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="bg-pink-50 hover:bg-pink-100 p-4 rounded-lg border border-pink-200 transition-colors"
                  >
                    <p className="text-sm text-gray-600 mb-2">إنستجرام</p>
                    <p className="text-pink-600 font-semibold truncate">
                      الملف الشخصي
                    </p>
                  </a>
                )}
                {data.linkedin && (
                  <a
                    href={data.linkedin}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="bg-blue-50 hover:bg-blue-100 p-4 rounded-lg border border-blue-200 transition-colors"
                  >
                    <p className="text-sm text-gray-600 mb-2">لينكدإن</p>
                    <p className="text-blue-700 font-semibold truncate">
                      الملف الشخصي
                    </p>
                  </a>
                )}
              </div>
            </InfoCard>
          )}

          {/* Documents */}
          {data.documents && data.documents.length > 0 && (
            <InfoCard title="المستندات المرفقة" icon={FileText}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {data.documents.map((doc: any, idx: number) => (
                  <div
                    key={idx}
                    className="bg-gradient-to-br from-gray-50 to-slate-50 p-5 rounded-lg border border-gray-300"
                  >
                    <div className="flex items-start justify-between mb-3">
                      <div>
                        <p className="text-sm text-gray-600 mb-1">
                          نوع المستند
                        </p>
                        <p className="text-lg font-bold text-gray-800">
                          {getDocumentTypeLabel(doc.type)}
                        </p>
                      </div>
                      <FileText className="w-8 h-8 text-gray-400" />
                    </div>
                    <div className="mt-3 pt-3 border-t border-gray-200">
                      <p className="text-xs text-gray-500">
                        تاريخ الرفع:{" "}
                        {new Date(doc.created_at).toLocaleDateString("ar-EG")}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </InfoCard>
          )}

          {/* Footer Actions */}
          <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border-2 border-blue-200 rounded-2xl p-8 shadow-lg">
            <div className="text-center mb-6">
              <Calendar className="w-12 h-12 text-blue-600 mx-auto mb-4" />
              <p className="text-gray-700 mb-2 text-lg font-semibold">
                تم استلام طلبك بنجاح
              </p>
              <p className="text-sm text-gray-600">
                تاريخ التقديم:{" "}
                {new Date(data.created_at).toLocaleDateString("ar-EG", {
                  year: "numeric",
                  month: "long",
                  day: "numeric",
                  hour: "2-digit",
                  minute: "2-digit",
                })}
              </p>
            </div>

            <div className="flex justify-center gap-4 flex-wrap mb-6">
              <button
                onClick={() => window.print()}
                className="flex items-center gap-2 bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-semibold transition-colors shadow-md"
              >
                <Download className="w-5 h-5" />
                طباعة / تحميل البيانات
              </button>
            </div>

            <div className="p-4 bg-green-100 border border-green-300 rounded-lg text-center">
              <p className="text-sm text-green-800 font-semibold">
                ✓ تم قفل النموذج بعد الإرسال الناجح. سيتم التواصل معك قريباً.
              </p>
            </div>
          </div>

          {/* Print Styles */}
          <style jsx>{`
            @media print {
              body {
                background: white;
              }
              button {
                display: none;
              }
            }
          `}</style>
        </div>
      </div>
    );
  }

  const currentSectionData = config.sections[currentSection];

  // Handle step click with validation
  // Handle step click with proper validation
  const handleStepClick = async (stepIndex: number) => {
    // Can only go to previous sections or stay in current
    if (stepIndex > currentSection) {
      return;
    }

    // If going backwards, just go (no validation needed for previous sections)
    if (stepIndex < currentSection) {
      setCurrentSection(stepIndex);
      window.scrollTo({ top: 0, behavior: "smooth" });
      return;
    }
  };

  return (
    <form onSubmit={handleSubmit(onSubmit)} className="max-w-6xl mx-auto">
      <ProgressBar
        currentSection={currentSection}
        totalSections={config.sections.length}
        sectionTitle={currentSectionData.title}
        sections={config.sections}
        onStepClick={handleStepClick}
        validSections={config.sections.map((_, sectionIndex) =>
          isSectionValidByIndex(sectionIndex)
        )}
      />

      <div className="bg-white rounded-xl shadow-lg p-8 mb-6">
        <h2 className="text-2xl font-bold text-gray-800 mb-2 border-b pb-4">
          {currentSectionData.title}
        </h2>
        {currentSectionData.description && (
          <p className="text-gray-600 mb-6">{currentSectionData.description}</p>
        )}

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {currentSectionData.fields.map((field) => (
            <div
              key={field.name}
              className={
                field.type === "textarea" ||
                field.type === "file" ||
                field.type === "array" ||
                field.name === "documents"
                  ? "md:col-span-2"
                  : ""
              }
            >
              <FormField
                field={field}
                register={register}
                errors={errors}
                watch={watch}
                setValue={setValue}
                control={control}
              />
            </div>
          ))}
        </div>
      </div>

      {submitError && (
        <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded mb-6">
          <p className="font-semibold">خطأ:</p>
          <p>{submitError}</p>
        </div>
      )}

      <div className="flex justify-between items-center">
        <Button
          type="button"
          variant="outline"
          onClick={prevSection}
          disabled={currentSection === 0}
        >
          <ChevronRight className="w-5 h-5" />
          السابق
        </Button>

        {currentSection < config.sections.length - 1 ? (
          <Button
            type="button"
            onClick={nextSection}
            disabled={!isSectionValid() || isValidating}
          >
            التالي
            <ChevronLeft className="w-5 h-5" />
          </Button>
        ) : (
          <Button type="submit" isLoading={isSubmitting}>
            <Send className="w-5 h-5" />
            تقديم الاستمارة
          </Button>
        )}
      </div>
    </form>
  );
}
