"use client";

import React, { useState, useEffect } from "react";
import { useFieldArray } from "react-hook-form";
import { FormField as FormFieldType } from "@/lib/types";
import { Button } from "@/components/ui/Button";
import { Plus, Trash2 } from "lucide-react";
import { Select } from "@/components/ui/Select";
import { Input } from "@/components/ui/Input";
import { apiCall } from "@/lib/utils";

// Global persistent cache for options
const optionsCacheArray = new Map<
  string,
  { data: any[]; fetching: Promise<any[]> | null }
>();

interface ArrayFieldInputProps {
  field: FormFieldType;
  control: any;
  errors: any;
  watch: any;
  register: any;
}

export function ArrayFieldInput({
  field,
  control,
  errors,
  watch,
  register,
}: ArrayFieldInputProps) {
  const { fields, append, remove } = useFieldArray({
    control,
    name: field.name,
  });

  // State to store dynamic options for each array field
  const [dynamicOptions, setDynamicOptions] = useState<Record<string, any[]>>(
    {}
  );
  const [loadingOptions, setLoadingOptions] = useState<Record<string, boolean>>(
    {}
  );

  // Fetch dynamic options for array fields - only once with persistent caching
  useEffect(() => {
    field.arrayFields?.forEach((arrayField) => {
      if (!arrayField.optionsEndpoint) return;

      // Check cache first - if already cached, use it immediately
      if (optionsCacheArray.has(arrayField.optionsEndpoint)) {
        const cached = optionsCacheArray.get(arrayField.optionsEndpoint)!;
        setDynamicOptions((prev) => ({
          ...prev,
          [arrayField.name]: cached.data,
        }));

        // If currently fetching, wait for it
        if (cached.fetching) {
          cached.fetching.then((options) => {
            setDynamicOptions((prev) => ({
              ...prev,
              [arrayField.name]: options,
            }));
          });
        }
        return;
      }

      // Not in cache, start fetching
      setLoadingOptions((prev) => ({ ...prev, [arrayField.name]: true }));

      const fetchPromise = apiCall(arrayField.optionsEndpoint)
        .then((res) => res.json())
        .then((data) => {
          const options = Array.isArray(data)
            ? data
            : data.data || data.options || [];
          setDynamicOptions((prev) => ({
            ...prev,
            [arrayField.name]: options,
          }));
          // Update cache with final data
          if (arrayField.optionsEndpoint) {
            const cacheEntry = optionsCacheArray.get(
              arrayField.optionsEndpoint
            );
            if (cacheEntry) {
              cacheEntry.data = options;
              cacheEntry.fetching = null;
            }
          }
          return options;
        })
        .catch((error) => {
          console.error(
            `Error fetching options for ${arrayField.name}:`,
            error
          );
          setDynamicOptions((prev) => ({
            ...prev,
            [arrayField.name]: [],
          }));
          return [];
        })
        .finally(() => {
          setLoadingOptions((prev) => ({
            ...prev,
            [arrayField.name]: false,
          }));
        });

      // Store in cache with fetching promise
      optionsCacheArray.set(arrayField.optionsEndpoint, {
        data: [],
        fetching: fetchPromise,
      });
    });
  }, [field.arrayFields]);

  const handleAdd = () => {
    const newItem: any = {};
    field.arrayFields?.forEach((arrayField) => {
      newItem[arrayField.name] = "";
    });
    append(newItem);
  };

  return (
    <div className="border rounded-lg p-4 bg-gray-50">
      <div className="flex justify-between items-center mb-4">
        <label className="block text-sm font-medium text-gray-700">
          {field.label}
          {field.required && <span className="text-red-500 ml-1">*</span>}
        </label>
        <Button
          type="button"
          variant="outline"
          onClick={handleAdd}
          className="text-sm"
        >
          <Plus className="w-4 h-4" />
          إضافة
        </Button>
      </div>

      {fields.length === 0 && (
        <p className="text-sm text-gray-500 text-center py-4">
          لم يتم إضافة عناصر بعد. اضغط "إضافة" لإضافة عنصر.
        </p>
      )}

      <div className="space-y-4">
        {fields.map((item, index) => (
          <div
            key={item.id}
            className="bg-white p-4 rounded-lg border border-gray-200"
          >
            <div className="flex justify-between items-center mb-3">
              <h4 className="text-sm font-medium text-gray-700">
                العنصر {index + 1}
              </h4>
              <button
                type="button"
                onClick={() => remove(index)}
                className="text-red-500 hover:text-red-700"
              >
                <Trash2 className="w-4 h-4" />
              </button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {field.arrayFields?.map((arrayField) => {
                const fieldName = `${field.name}.${index}.${arrayField.name}`;
                const fieldError =
                  errors[field.name]?.[index]?.[arrayField.name];

                if (arrayField.type === "select") {
                  // Use dynamic options if endpoint is provided, otherwise use static options
                  const options = arrayField.optionsEndpoint
                    ? dynamicOptions[arrayField.name] || []
                    : arrayField.options || [];
                  const optionStrings = options.map((opt) =>
                    typeof opt === "string" ? opt : opt.label
                  );
                  const optionValues = options.map((opt) =>
                    typeof opt === "string" ? opt : String(opt.value || opt.id)
                  );
                  const isLoading = loadingOptions[arrayField.name] || false;

                  return (
                    <Select
                      key={fieldName}
                      label={arrayField.label}
                      required={arrayField.required}
                      options={optionStrings}
                      optionValues={optionValues}
                      error={fieldError?.message}
                      disabled={isLoading || options.length === 0}
                      {...register(fieldName)}
                    />
                  );
                }

                return (
                  <Input
                    key={fieldName}
                    type={arrayField.type}
                    label={arrayField.label}
                    required={arrayField.required}
                    error={fieldError?.message}
                    {...register(fieldName)}
                  />
                );
              })}
            </div>
          </div>
        ))}
      </div>

      {errors[field.name]?.message && (
        <p className="mt-2 text-sm text-red-500">
          {errors[field.name]?.message}
        </p>
      )}
    </div>
  );
}
