"use client";

import { useState, useEffect } from "react";
import {
  CheckCircle,
  XCircle,
  Clock,
  BarChart3,
  Download,
  ChevronDown,
} from "lucide-react";

interface ExamResult {
  id?: number;
  exam_id?: number;
  exam_title?: string;
  score?: string | number;
  answers?: Record<string, string>;
  started_at?: string;
  completed_at?: string;
  passing_score?: number;
  auto_publish_results?: boolean;
  exam?: {
    id: number;
    title: string;
    passing_score: number;
    form_schema?: any;
  };
  questions?: any[];
  // New structure from admin endpoint
  attempt?: {
    id: number;
    exam_id: number;
    exam_title: string;
    score: string | number;
    passing_score: number;
    started_at: string;
    completed_at: string;
    auto_submitted: boolean;
  };
  results?: {
    score: number;
    passing_score: number;
    passed: boolean;
    total_questions: number;
    correct_answers: number;
    wrong_answers: number;
  };
}

interface ExamResultsDisplayProps {
  result: ExamResult;
  onDownloadDocument?: (docId: number, docType: string) => Promise<void>;
  showDetailedView?: boolean;
  detailedResults?: any;
  isLoading?: boolean;
}

export function ExamResultsDisplay({
  result,
  onDownloadDocument,
  showDetailedView = false,
  detailedResults,
  isLoading = false,
}: ExamResultsDisplayProps) {
  const [expandedQuestions, setExpandedQuestions] = useState<Set<number>>(
    new Set()
  );
  const [isInitialized, setIsInitialized] = useState(false);

  // Handle both data structures (old and new)
  const attempt = result.attempt || result;
  const rawScore = attempt.score || result.score || 0;
  const score =
    typeof rawScore === "string" ? parseFloat(rawScore) : Number(rawScore);

  const passingScore =
    attempt.passing_score ||
    result.passing_score ||
    result.exam?.passing_score ||
    60;
  const isPassed = score >= passingScore;

  // Get questions from detailedResults or fallback to result.questions or calculate from answers
  const questions = detailedResults?.questions || result.questions || [];
  const totalQuestions =
    questions.length || Object.keys(result.answers || {}).length || 0;
  const correctAnswers =
    questions.filter((q: any) => q.is_correct).length ||
    result.results?.correct_answers ||
    0;
  const wrongAnswers = totalQuestions - correctAnswers;

  const startedAt = attempt.started_at || result.started_at || "";
  const completedAt = attempt.completed_at || result.completed_at || "";

  // Initialize all questions as expanded on first load only
  useEffect(() => {
    if (questions.length > 0 && !isInitialized) {
      const allIndices = new Set<number>(
        questions.map((_: any, index: number) => index)
      );
      setExpandedQuestions(allIndices);
      setIsInitialized(true);
    }
  }, [questions, isInitialized]);

  const formatDuration = (startedAt: string, completedAt: string) => {
    if (!startedAt || !completedAt) return "N/A";
    const start = new Date(startedAt).getTime();
    const end = new Date(completedAt).getTime();
    const durationMs = end - start;
    const minutes = Math.floor(durationMs / 60000);
    const seconds = Math.floor((durationMs % 60000) / 1000);
    return `${minutes}د ${seconds}ث`;
  };

  return (
    <div className="space-y-6">
      {/* Results Summary Card */}
      <div
        className={`rounded-lg shadow-lg p-8 text-white ${
          isPassed
            ? "bg-gradient-to-r from-green-600 to-emerald-600"
            : "bg-gradient-to-r from-red-600 to-pink-600"
        }`}
      >
        <div className="flex items-center justify-between mb-6">
          <div>
            <h2 className="text-3xl font-bold mb-2">
              {attempt.exam_title || result.exam?.title || "امتحان"}
            </h2>
            <p className="text-green-100">الامتحان اكتمل بنجاح</p>
          </div>
          {isPassed ? (
            <CheckCircle className="w-16 h-16" />
          ) : (
            <XCircle className="w-16 h-16" />
          )}
        </div>

        <div className="grid grid-cols-2 md:grid-cols-4 text-black gap-4 mt-8">
          <div className="bg-white bg-opacity-20 rounded-lg p-4">
            <p className="text-sm opacity-90">النتيجة</p>
            <p className="text-3xl font-bold">{score.toFixed(2)}</p>
          </div>
          <div className="bg-white bg-opacity-20 rounded-lg p-4">
            <p className="text-sm opacity-90">درجة النجاح</p>
            <p className="text-3xl font-bold">{passingScore}</p>
          </div>
          <div className="bg-white bg-opacity-20 rounded-lg p-4">
            <p className="text-sm opacity-90">النسبة المئوية</p>
            <p className="text-3xl font-bold">
              {((score / passingScore) * 100).toFixed(0)}%
            </p>
          </div>
          <div className="bg-white bg-opacity-20 rounded-lg p-4">
            <p className="text-sm opacity-90">الحالة</p>
            <p className="text-2xl font-bold">{isPassed ? "نجح" : "فشل"}</p>
          </div>
        </div>
      </div>

      {/* Statistics Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <div className="bg-blue-50 rounded-lg p-6 border-l-4 border-blue-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm mb-1">عدد الأسئلة</p>
              <p className="text-3xl font-bold text-blue-600">
                {totalQuestions}
              </p>
            </div>
            <BarChart3 className="w-8 h-8 text-blue-500 opacity-50" />
          </div>
        </div>

        <div className="bg-green-50 rounded-lg p-6 border-l-4 border-green-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm mb-1">الإجابات الصحيحة</p>
              <p className="text-3xl font-bold text-green-600">
                {correctAnswers}
              </p>
            </div>
            <CheckCircle className="w-8 h-8 text-green-500 opacity-50" />
          </div>
        </div>

        <div className="bg-red-50 rounded-lg p-6 border-l-4 border-red-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm mb-1">الإجابات الخاطئة</p>
              <p className="text-3xl font-bold text-red-600">{wrongAnswers}</p>
            </div>
            <XCircle className="w-8 h-8 text-red-500 opacity-50" />
          </div>
        </div>

        <div className="bg-purple-50 rounded-lg p-6 border-l-4 border-purple-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm mb-1">الوقت المستغرق</p>
              <p className="text-lg font-bold text-purple-600">
                {formatDuration(startedAt, completedAt)}
              </p>
            </div>
            <Clock className="w-8 h-8 text-purple-500 opacity-50" />
          </div>
        </div>
      </div>

      {/* Detailed Results */}
      {showDetailedView && (detailedResults?.questions || result.questions) && (
        <div className="bg-white rounded-lg shadow-md overflow-hidden">
          <div className="p-6 border-b-2 border-gray-200">
            <div className="flex items-center justify-between mb-0">
              <h3 className="text-2xl font-bold text-gray-800">
                عرض مفصل للإجابات
              </h3>
              <div className="flex gap-2">
                <button
                  onClick={() => {
                    const allIndices = new Set<number>(
                      questions.map((_: any, index: number) => index)
                    );
                    setExpandedQuestions(allIndices);
                  }}
                  className="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-sm font-medium transition-colors"
                >
                  فتح الكل
                </button>
                <button
                  onClick={() => {
                    setExpandedQuestions(new Set<number>());
                  }}
                  className="px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white rounded-lg text-sm font-medium transition-colors"
                >
                  إغلاق الكل
                </button>
              </div>
            </div>
          </div>

          {isLoading ? (
            <div className="p-6 text-center">
              <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-blue-500 mx-auto"></div>
              <p className="text-gray-600 mt-2">جاري تحميل التفاصيل...</p>
            </div>
          ) : (
            <div className="space-y-4 p-6">
              {questions.map((question: any, index: number) => (
                <div
                  key={question.id || index}
                  className={`border-l-4 rounded-lg p-4 transition ${
                    question.is_correct
                      ? "border-green-500 bg-green-50"
                      : "border-red-500 bg-red-50"
                  }`}
                >
                  <div
                    className="flex items-start justify-between cursor-pointer hover:opacity-80 transition"
                    onClick={() => {
                      const newExpanded = new Set(expandedQuestions);
                      if (newExpanded.has(index)) {
                        newExpanded.delete(index);
                      } else {
                        newExpanded.add(index);
                      }
                      setExpandedQuestions(newExpanded);
                    }}
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-2">
                        <span className="text-sm font-semibold text-gray-600">
                          سؤال {index + 1}
                        </span>
                        {question.is_correct ? (
                          <CheckCircle className="w-5 h-5 text-green-600" />
                        ) : (
                          <XCircle className="w-5 h-5 text-red-600" />
                        )}
                      </div>
                      <p className="text-gray-800 font-medium mb-2">
                        {question.text || question.question}
                      </p>
                      <p className="text-xs text-gray-600">
                        النوع: {getQuestionTypeLabel(question.type)}
                      </p>
                    </div>
                    <div className="flex items-start gap-3 ml-4">
                      <div
                        className={`text-sm font-bold px-3 py-1 rounded ${
                          question.is_correct
                            ? "bg-green-200 text-green-800"
                            : "bg-red-200 text-red-800"
                        }`}
                      >
                        {question.is_correct ? "صحيح" : "خطأ"}
                      </div>
                      <ChevronDown
                        className={`w-5 h-5 text-gray-600 flex-shrink-0 transition-transform ${
                          expandedQuestions.has(index) ? "rotate-180" : ""
                        }`}
                      />
                    </div>
                  </div>

                  {expandedQuestions.has(index) && (
                    <div className="mt-4 pt-4 border-t-2 border-gray-200 space-y-3">
                      {/* User's Answer */}
                      <div className="bg-white p-3 rounded">
                        <p className="text-sm text-gray-600 mb-2 font-semibold">
                          إجابتك:
                        </p>
                        <div
                          className={`p-3 rounded ${
                            question.is_correct
                              ? "bg-green-100 border border-green-300"
                              : "bg-red-100 border border-red-300"
                          }`}
                        >
                          <p className="text-gray-800">
                            {question.user_answer || "لم تجب"}
                          </p>
                        </div>
                      </div>

                      {/* Correct Answer */}
                      {!question.is_correct && (
                        <div className="bg-white p-3 rounded">
                          <p className="text-sm text-gray-600 mb-2 font-semibold">
                            الإجابة الصحيحة:
                          </p>
                          <div className="p-3 rounded bg-green-100 border border-green-300">
                            <p className="text-gray-800">
                              {question.correct_answer}
                            </p>
                          </div>
                        </div>
                      )}

                      {/* Options for Multiple Choice */}
                      {question.type === "multiple_choice" &&
                        question.options && (
                          <div className="bg-white p-3 rounded">
                            <p className="text-sm text-gray-600 mb-2 font-semibold">
                              الخيارات:
                            </p>
                            <div className="space-y-2">
                              {question.options.map(
                                (option: string, idx: number) => (
                                  <div
                                    key={idx}
                                    className={`p-2 rounded text-sm ${
                                      option === question.correct_answer
                                        ? "bg-green-100 border border-green-300 text-green-900"
                                        : option === question.user_answer
                                        ? "bg-red-100 border border-red-300 text-red-900"
                                        : "bg-gray-100 border border-gray-300 text-gray-900"
                                    }`}
                                  >
                                    {option}
                                  </div>
                                )
                              )}
                            </div>
                          </div>
                        )}
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </div>
      )}

      {/* Summary Footer */}
      <div className="bg-gray-50 rounded-lg p-4 flex items-center justify-between">
        <div>
          <p className="text-sm text-gray-600">بدء الامتحان</p>
          <p className="font-medium text-gray-800">
            {startedAt ? new Date(startedAt).toLocaleString("ar-SA") : "N/A"}
          </p>
        </div>
        <div>
          <p className="text-sm text-gray-600">انتهاء الامتحان</p>
          <p className="font-medium text-gray-800">
            {completedAt
              ? new Date(completedAt).toLocaleString("ar-SA")
              : "N/A"}
          </p>
        </div>
      </div>
    </div>
  );
}

function getQuestionTypeLabel(type: string): string {
  const labels: Record<string, string> = {
    multiple_choice: "اختيار من متعدد",
    true_false: "صح/خطأ",
    essay: "مقالي",
  };
  return labels[type] || type;
}
