"use client";

import { useState, useEffect } from "react";
import { Download, Eye, FileText, Image, File } from "lucide-react";

interface Document {
  id: number;
  type: string;
  file_path: string;
}

interface DocumentsGridProps {
  documents: Document[];
  userId?: number;
  judgeMode?: boolean;
}

const documentTypeLabels: Record<string, string> = {
  national_id: "الهوية القومية",
  cv: "السيرة الذاتية",
  photo_official: "صورة رسمية",
  photo_casual: "صورة شخصية",
  degree_certificate: "شهادة التخرج",
  passport: "جواز السفر",
  driving_license: "رخصة القيادة",
  birth_certificate: "شهادة الميلاد",
};

const getDocumentIcon = (filePath: string) => {
  if (filePath.match(/\.(jpg|jpeg|png|gif|webp)$/i)) {
    return <Image className="w-5 h-5" />;
  }
  if (filePath.match(/\.pdf$/i)) {
    return <File className="w-5 h-5" />;
  }
  return <FileText className="w-5 h-5" />;
};

const getFileExtension = (filePath: string): string => {
  const parts = filePath.split(".");
  return parts[parts.length - 1].toUpperCase();
};

export default function DocumentsGrid({
  documents,
  userId,
  judgeMode = false,
}: DocumentsGridProps) {
  const [selectedDoc, setSelectedDoc] = useState<Document | null>(null);
  const [downloadLoading, setDownloadLoading] = useState<number | null>(null);
  const [previewData, setPreviewData] = useState<{
    id: number;
    dataUrl: string;
    type: "image" | "pdf" | "unsupported";
  } | null>(null);
  const [thumbnails, setThumbnails] = useState<Record<number, string>>({});
  const [loadingThumbnails, setLoadingThumbnails] = useState<Set<number>>(
    new Set()
  );

  // Build the document URL based on whether it's judge mode
  const getDocumentUrl = (docId: number, action: "preview" | "download") => {
    const baseUrl = process.env.NEXT_PUBLIC_API_URL;
    if (judgeMode && userId) {
      return `${baseUrl}/judge/users/${userId}/documents/${docId}/${action}`;
    }
    return `${baseUrl}/documents/${docId}/${action}`;
  };

  useEffect(() => {
    // Load thumbnails for all image documents
    documents.forEach((doc) => {
      if (doc.file_path.match(/\.(jpg|jpeg|png|gif|webp)$/i)) {
        loadThumbnail(doc);
      }
    });
  }, [documents]);

  const loadThumbnail = async (doc: Document) => {
    if (thumbnails[doc.id]) return; // Already loaded

    setLoadingThumbnails((prev) => new Set(prev).add(doc.id));
    try {
      const response = await fetch(getDocumentUrl(doc.id, "preview"), {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("auth_token")}`,
        },
      });

      if (response.ok) {
        const blob = await response.blob();
        const dataUrl = URL.createObjectURL(blob);
        setThumbnails((prev) => ({ ...prev, [doc.id]: dataUrl }));
      }
    } catch (error) {
      console.error(`Failed to load thumbnail for document ${doc.id}:`, error);
    } finally {
      setLoadingThumbnails((prev) => {
        const updated = new Set(prev);
        updated.delete(doc.id);
        return updated;
      });
    }
  };

  const handlePreview = async (doc: Document) => {
    setSelectedDoc(doc);

    try {
      const response = await fetch(getDocumentUrl(doc.id, "preview"), {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("auth_token")}`,
        },
      });

      if (!response.ok) {
        throw new Error("Failed to load preview");
      }

      const blob = await response.blob();
      const isImage = doc.file_path.match(/\.(jpg|jpeg|png|gif|webp)$/i);
      const isPdf = doc.file_path.match(/\.pdf$/i);

      if (isImage) {
        const dataUrl = URL.createObjectURL(blob);
        setPreviewData({ id: doc.id, dataUrl, type: "image" });
      } else if (isPdf) {
        const dataUrl = URL.createObjectURL(blob);
        setPreviewData({ id: doc.id, dataUrl, type: "pdf" });
      } else {
        setPreviewData({ id: doc.id, dataUrl: "", type: "unsupported" });
      }
    } catch (error) {
      console.error("Preview failed:", error);
      alert("فشل تحميل معاينة المستند");
      setSelectedDoc(null);
    }
  };

  const handleDownload = async (doc: Document) => {
    setDownloadLoading(doc.id);
    try {
      const response = await fetch(getDocumentUrl(doc.id, "download"), {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("auth_token")}`,
        },
      });

      if (!response.ok) {
        throw new Error("Failed to download document");
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement("a");
      link.href = url;
      link.download = `${doc.type}_${doc.id}.${getFileExtension(
        doc.file_path
      ).toLowerCase()}`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);
    } catch (error) {
      console.error("Download failed:", error);
      alert("فشل تحميل المستند");
    } finally {
      setDownloadLoading(null);
    }
  };

  useEffect(() => {
    return () => {
      if (previewData?.dataUrl) {
        URL.revokeObjectURL(previewData.dataUrl);
      }
      // Clean up thumbnail URLs
      Object.values(thumbnails).forEach((url) => {
        URL.revokeObjectURL(url);
      });
    };
  }, [previewData, thumbnails]);

  if (documents.length === 0) {
    return (
      <div className="text-center py-12 bg-gray-50 rounded-lg">
        <FileText className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500">لا توجد مستندات</p>
      </div>
    );
  }

  return (
    <>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {documents.map((doc) => (
          <div
            key={doc.id}
            className="bg-white border border-gray-200 rounded-lg overflow-hidden hover:shadow-lg transition-shadow flex flex-col"
          >
            {/* Thumbnail Preview */}
            <div className="relative bg-gray-100 aspect-video overflow-hidden flex items-center justify-center">
              {thumbnails[doc.id] ? (
                <img
                  src={thumbnails[doc.id]}
                  alt={documentTypeLabels[doc.type] || doc.type}
                  className="w-full h-full object-cover"
                />
              ) : doc.file_path.match(/\.(jpg|jpeg|png|gif|webp)$/i) ? (
                loadingThumbnails.has(doc.id) ? (
                  <div className="flex items-center justify-center">
                    <div className="w-8 h-8 border-4 border-gray-300 border-t-blue-600 rounded-full animate-spin" />
                  </div>
                ) : (
                  <div className="flex flex-col items-center justify-center text-gray-400">
                    <Image className="w-12 h-12 mb-2" />
                    <p className="text-xs">جاري التحميل...</p>
                  </div>
                )
              ) : (
                <div className="flex flex-col items-center justify-center text-gray-400">
                  {getDocumentIcon(doc.file_path)}
                  <p className="text-xs mt-2">
                    {getFileExtension(doc.file_path)}
                  </p>
                </div>
              )}
            </div>

            {/* Document Info */}
            <div className="p-4 flex-1 flex flex-col">
              <p className="font-semibold text-gray-900 text-sm truncate">
                {documentTypeLabels[doc.type] || doc.type}
              </p>

              {/* Action Buttons */}
              <div className="flex gap-2 mt-auto pt-3">
                <button
                  onClick={() => handlePreview(doc)}
                  className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-blue-50 hover:bg-blue-100 text-blue-600 rounded-lg transition-colors text-sm font-medium"
                >
                  <Eye className="w-4 h-4" />
                  <span>عرض</span>
                </button>
                <button
                  onClick={() => handleDownload(doc)}
                  disabled={downloadLoading === doc.id}
                  className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-green-50 hover:bg-green-100 text-green-600 rounded-lg transition-colors text-sm font-medium disabled:opacity-50"
                >
                  <Download className="w-4 h-4" />
                  <span>
                    {downloadLoading === doc.id ? "جاري..." : "تحميل"}
                  </span>
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Preview Modal */}
      {selectedDoc && (
        <div
          className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4"
          onClick={() => {
            setSelectedDoc(null);
            setPreviewData(null);
          }}
        >
          <div
            className="bg-white rounded-xl max-w-4xl w-full max-h-[90vh] overflow-auto p-6"
            onClick={(e) => e.stopPropagation()}
          >
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">
                {documentTypeLabels[selectedDoc.type] || selectedDoc.type}
              </h3>
              <button
                onClick={() => {
                  setSelectedDoc(null);
                  setPreviewData(null);
                }}
                className="text-gray-500 hover:text-gray-700 text-2xl"
              >
                ×
              </button>
            </div>

            <div className="bg-gray-50 rounded-lg overflow-auto">
              {previewData?.type === "image" ? (
                <img
                  src={previewData.dataUrl}
                  alt={selectedDoc.type}
                  className="w-full h-auto"
                />
              ) : previewData?.type === "pdf" ? (
                <iframe
                  src={previewData.dataUrl}
                  className="w-full h-[600px]"
                  title={selectedDoc.type}
                />
              ) : (
                <div className="flex items-center justify-center h-64">
                  <div className="text-center">
                    <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                    <p className="text-gray-600 mb-4">جاري تحميل المعاينة...</p>
                  </div>
                </div>
              )}
            </div>

            <div className="flex gap-3 mt-4">
              <button
                onClick={() => handleDownload(selectedDoc)}
                disabled={downloadLoading === selectedDoc.id}
                className="flex-1 flex items-center justify-center gap-2 px-4 py-3 bg-green-600 hover:bg-green-700 text-white rounded-lg transition-colors disabled:opacity-50"
              >
                <Download className="w-5 h-5" />
                {downloadLoading === selectedDoc.id
                  ? "جاري التحميل..."
                  : "تحميل"}
              </button>
              <button
                onClick={() => {
                  setSelectedDoc(null);
                  setPreviewData(null);
                }}
                className="flex-1 px-4 py-3 bg-gray-200 hover:bg-gray-300 text-gray-700 rounded-lg transition-colors"
              >
                إغلاق
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
}
