"use client";

import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { RegistrationGuard } from "@/components/form/RegistrationGuard";
import { StructuredData } from "@/components/seo/StructuredData";
import { generateWebPageSchema } from "@/lib/seo";
import formConfig from "@/public/formConfig.json";
import Image from "next/image";
import cover from "@/public/cover.jpeg";
import axios from "axios";

export default function Home() {
  const router = useRouter();
  const [registrationOpen, setRegistrationOpen] = useState(true);
  const [loading, setLoading] = useState(true);

  const DOMAIN = process.env.NEXT_PUBLIC_DOMAIN || "https://your-domain.com";
  const pageSchema = generateWebPageSchema(
    "استمارة التقدم لبرنامج الدبلوماسية الشبابية",
    "استمارة التقدم الرسمية للتسجيل في برنامج الدبلوماسية الشبابية التابع لوزارة الشباب والرياضة",
    DOMAIN,
    "/og-image.png"
  );

  useEffect(() => {
    checkSystemStatus();
  }, []);

  const checkSystemStatus = async () => {
    try {
      const response = await axios.get(
        `${
          process.env.NEXT_PUBLIC_API_URL || "http://localhost:8000/api/v1"
        }/system-status`
      );

      // Check maintenance mode first
      if (response.data.maintenance_mode) {
        router.push("/maintenance");
        return;
      }

      setRegistrationOpen(response.data.registration_open);
    } catch (error: any) {
      console.error("Failed to check system status:", error);

      // If maintenance mode error (503), redirect
      if (
        error.response?.data?.maintenance_mode ||
        error.response?.status === 503
      ) {
        router.push("/maintenance");
        return;
      }

      // Default to open on other errors
      setRegistrationOpen(true);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <main className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="text-gray-600 mt-4">جاري التحميل...</p>
        </div>
      </main>
    );
  }

  if (!registrationOpen) {
    return (
      <main className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center px-4">
        <div
          className="max-w-2xl mx-auto text-center bg-white rounded-lg shadow-xl p-12"
          dir="rtl"
        >
          <div className="mb-8">
            <svg
              className="w-24 h-24 mx-auto text-gray-400"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"
              />
            </svg>
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-4">شكراً</h1>
          <p className="text-2xl text-gray-700 mb-8">
            لقد تم انتهاء مدة التسجيل
          </p>
          <p className="text-gray-600">
            نشكرك على اهتمامك ببرنامج الدبلوماسية الشبابية. لقد انتهت فترة
            التسجيل حالياً. يرجى متابعة موقعنا الإلكتروني للحصول على تحديثات حول
            فترات التسجيل القادمة.
          </p>
        </div>
      </main>
    );
  }

  return (
    <main className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4">
      <StructuredData data={pageSchema} id="home-page-schema" />

      <div className="max-w-7xl mx-auto">
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            {formConfig.formTitle}
          </h1>
          <Image
            src={cover}
            alt="برنامج الدبلوماسية الشبابية - استمارة التقدم"
            className="mb-4"
            priority
          />
          <p className="text-gray-600 leading-relaxed">
            تخضع هذه الاستمارة الي التقييم والمراجعة ومن ثم التصفية للانتقال
            لمرحلة المقابلة الشخصية والاختبار الشفهي والتحريري في مبادئ واساسيات
            العلوم السياسية وتاريخ مصر والجغرافيا والقانون الدولي والاقتصاد
            والعلاقات الدولية والمنظمات الدولية واللغات العربية والانجليزية
            بجانب اللغة الثالثة، اقر بالتسجيل علي هذه الاستمارة بالموافقة علي
            التقدم للمقابلات الشخصية وكافة الاختبارات والتقييمات التي تعدها
            وزارة الشباب والرياضة لقبولي في برنامج الدبلوماسية الشبابية، كما اني
            ملتزم بحضور كافة الدورات التدريبية وبرامج بناء القدرات الخاصة
            بالبرنامج.
          </p>
        </div>

        <RegistrationGuard config={formConfig} />
      </div>
    </main>
  );
}
