"use client";

import { useEffect, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { judgeAPI, handleAPIError } from "@/lib/api";
import {
  User,
  BarChart3,
  AlertCircle,
  Search,
  ChevronLeft,
  ChevronRight,
} from "lucide-react";

interface AssignedUser {
  id: number;
  name: string;
  email: string;
  status: string;
  evaluations_count: number;
  created_at: string;
}

interface PaginationData {
  total: number;
  count: number;
  per_page: number;
  current_page: number;
  last_page: number;
  from: number;
  to: number;
}

export default function JudgeUsersPage() {
  const router = useRouter();
  const searchParams = useSearchParams();

  const [users, setUsers] = useState<AssignedUser[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState(searchParams.get("search") || "");
  const [pagination, setPagination] = useState<PaginationData | null>(null);
  const [currentPage, setCurrentPage] = useState(
    Number(searchParams.get("page")) || 1
  );
  const [formalPhotos, setFormalPhotos] = useState<Record<number, string>>({});

  const perPage = 12;

  useEffect(() => {
    loadUsers(currentPage, search);
  }, [currentPage, search]);

  const loadUsers = async (page: number, searchQuery: string) => {
    try {
      setLoading(true);
      const response = await judgeAPI.getAssignedUsers({
        page,
        per_page: perPage,
        search: searchQuery || undefined,
      });

      const usersData = response.data.data.map((user: any) => ({
        id: user.id,
        name: user.name,
        email: user.email,
        status: user.status,
        evaluations_count: user.evaluations_count,
        created_at: user.created_at,
      }));

      setUsers(usersData);
      setPagination(response.data.pagination);

      // Load formal photos for all users
      const photoMap: Record<number, string> = {};
      for (const user of usersData) {
        try {
          const photoResponse = await fetch(
            `${process.env.NEXT_PUBLIC_API_URL}/judge/users/${user.id}/documents/photo_official/preview`,
            {
              headers: {
                Authorization: `Bearer ${localStorage.getItem("auth_token")}`,
              },
            }
          );

          if (photoResponse.ok) {
            const blob = await photoResponse.blob();
            const objectUrl = URL.createObjectURL(blob);
            photoMap[user.id] = objectUrl;
          }
        } catch (err) {
          console.error(`Failed to load photo for user ${user.id}`);
        }
      }

      setFormalPhotos(photoMap);

      // Update URL params
      const params = new URLSearchParams();
      if (searchQuery) params.set("search", searchQuery);
      params.set("page", page.toString());
      router.push(`/judge/users?${params.toString()}`);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const handleSearchChange = (value: string) => {
    setSearch(value);
    setCurrentPage(1);
  };

  const handlePageChange = (newPage: number) => {
    if (newPage >= 1 && newPage <= (pagination?.last_page || 1)) {
      setCurrentPage(newPage);
    }
  };

  if (loading && users.length === 0) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      <div>
        <h1 className="text-3xl font-bold text-gray-900">المتقدمين المعينين</h1>
        <p className="text-gray-600 mt-1">قم بتقييم المتقدمين المخصصين لك</p>
      </div>

      {/* Search */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="relative">
          <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
          <input
            type="text"
            value={search}
            onChange={(e) => handleSearchChange(e.target.value)}
            placeholder="البحث بالاسم أو البريد الإلكتروني..."
            className="w-full pr-11 pl-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>
      </div>

      {/* Users Grid */}
      {users.length === 0 ? (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <AlertCircle className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-500">
            {search ? "لا توجد نتائج للبحث" : "لا توجد متقدمين معينين"}
          </p>
        </div>
      ) : (
        <>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {users.map((user) => (
              <div
                key={user.id}
                className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow cursor-pointer"
                onClick={() => router.push(`/judge/users/${user.id}`)}
              >
                <div className="flex items-center gap-4 mb-4">
                  <div className="w-14 h-14 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-full flex items-center justify-center overflow-hidden flex-shrink-0">
                    {formalPhotos[user.id] ? (
                      <img
                        src={formalPhotos[user.id]}
                        alt={user.name}
                        className="w-full h-full object-cover"
                      />
                    ) : (
                      <User className="w-7 h-7 text-white" />
                    )}
                  </div>
                  <div className="flex-1">
                    <h3 className="font-semibold text-gray-900 mb-1">
                      {user.name}
                    </h3>
                    <p className="text-sm text-gray-500">{user.email}</p>
                  </div>
                </div>

                <div className="space-y-2">
                  <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <span className="text-sm text-gray-600">الحالة</span>
                    <span className="text-sm font-medium text-gray-900">
                      {user.status === "exam_completed" && "أكمل الاختبار"}
                      {user.status === "under_evaluation" && "تحت التقييم"}
                      {user.status === "evaluated" && "تم التقييم"}
                    </span>
                  </div>

                  <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                    <span className="text-sm text-gray-600">عدد التقييمات</span>
                    <span className="text-lg font-bold text-blue-600">
                      {user.evaluations_count}/5
                    </span>
                  </div>

                  {user.evaluations_count < 5 && (
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        router.push(`/judge/users/${user.id}/evaluate`);
                      }}
                      className="w-full py-3 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg hover:from-blue-700 hover:to-indigo-700 transition-colors font-medium"
                    >
                      تقييم الآن
                    </button>
                  )}
                </div>
              </div>
            ))}
          </div>

          {/* Pagination */}
          {pagination && pagination.last_page > 1 && (
            <div className="flex items-center justify-center gap-2 mt-8">
              <button
                onClick={() => handlePageChange(currentPage - 1)}
                disabled={currentPage === 1}
                className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                <ChevronLeft className="w-5 h-5" />
              </button>

              <div className="flex items-center gap-1">
                {Array.from(
                  { length: Math.min(5, pagination.last_page) },
                  (_, i) => {
                    let pageNum;
                    if (pagination.last_page <= 5) {
                      pageNum = i + 1;
                    } else if (currentPage <= 3) {
                      pageNum = i + 1;
                    } else if (currentPage >= pagination.last_page - 2) {
                      pageNum = pagination.last_page - 4 + i;
                    } else {
                      pageNum = currentPage - 2 + i;
                    }

                    return (
                      <button
                        key={pageNum}
                        onClick={() => handlePageChange(pageNum)}
                        className={`w-10 h-10 rounded-lg font-medium transition-colors ${
                          currentPage === pageNum
                            ? "bg-blue-600 text-white"
                            : "border border-gray-300 text-gray-900 hover:bg-gray-50"
                        }`}
                      >
                        {pageNum}
                      </button>
                    );
                  }
                )}
              </div>

              <button
                onClick={() => handlePageChange(currentPage + 1)}
                disabled={currentPage === pagination.last_page}
                className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                <ChevronRight className="w-5 h-5" />
              </button>

              <span className="text-sm text-gray-600 mr-4">
                صفحة {pagination.current_page} من {pagination.last_page}
              </span>
            </div>
          )}
        </>
      )}
    </div>
  );
}
