"use client";

import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { authAPI, handleAPIError } from "@/lib/api";
import {
  Mail,
  User,
  Phone,
  BarChart3,
  CheckCircle,
  AlertCircle,
  LogOut,
  MapPin,
  Briefcase,
  Globe,
  Download,
  FileText,
  Image as ImageIcon,
} from "lucide-react";

interface UserProfile {
  id: number;
  email: string;
  full_name: string;
  role: string;
  status: string;
  is_active: boolean;
  last_login_at: string | null;
  categories: string[];
  has_completed_exam: boolean;
  evaluation_count: number;
  is_evaluated: boolean;
}

interface ApplicantData {
  id: number;
  user_id: number;
  full_name_ar: string;
  full_name_en: string;
  national_id: string;
  passport_number: string;
  applicant_code: string;
  dob: string;
  gender: string;
  military_status: string;
  mobile: string;
  whatsapp: string;
  email: string;
  address: string;
  district: string;
  birth_place: string;
  other_nationality: string;
  degree: string;
  major: string;
  college: string;
  university: string;
  is_studying: boolean;
  study_year: string;
  grad_year: string;
  current_job: string;
  employer: string;
  experience_public: string;
  experience_ministry: string;
  experience_other_orgs: string;
  visited_countries: string;
  center_memberships: string;
  club_memberships: string;
  sports: string;
  skills_hobbies: string;
  motivation: string;
  strengths: string;
  weaknesses: string;
  facebook: string;
  instagram: string;
  linkedin: string;
  height: string;
  weight: string;
  governorate: {
    id: number;
    name: string;
  };
  nationalities: any[];
  languages: any[];
  degrees: any[];
  documents: any[];
}

function getDocumentLabel(type: string): string {
  const labels: Record<string, string> = {
    national_id: "بطاقة الهوية",
    passport: "جواز السفر",
    cv: "السيرة الذاتية",
    photo_official: "صورة رسمية",
    photo_casual: "صورة عادية",
    educational_certificate: "شهادة تعليمية",
    experience_certificate: "شهادة خبرة",
  };
  return labels[type] || type;
}

export default function ProfilePage() {
  const router = useRouter();
  const [profile, setProfile] = useState<UserProfile | null>(null);
  const [applicantData, setApplicantData] = useState<ApplicantData | null>(
    null
  );
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState("");
  const [successMessage, setSuccessMessage] = useState("");
  const [showPasswordForm, setShowPasswordForm] = useState(false);
  const [passwordForm, setPasswordForm] = useState({
    current_password: "",
    new_password: "",
    new_password_confirmation: "",
  });
  const [passwordLoading, setPasswordLoading] = useState(false);
  const [passwordError, setPasswordError] = useState("");
  const [downloadingDocId, setDownloadingDocId] = useState<number | null>(null);

  useEffect(() => {
    loadProfile();
  }, []);

  const loadProfile = async () => {
    try {
      setLoading(true);
      setError("");
      const [profileRes, applicantRes] = await Promise.allSettled([
        authAPI.getProfile(),
        authAPI.getApplicantData(),
      ]);

      if (profileRes.status === "fulfilled" && profileRes.value.data.success) {
        setProfile(profileRes.value.data.data);
      }

      if (
        applicantRes.status === "fulfilled" &&
        applicantRes.value.data.success
      ) {
        setApplicantData(applicantRes.value.data.data);
      }
    } catch (err) {
      setError(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const handleChangePassword = async (e: React.FormEvent) => {
    e.preventDefault();
    setPasswordError("");

    if (!passwordForm.new_password || !passwordForm.current_password) {
      setPasswordError("جميع الحقول مطلوبة");
      return;
    }

    if (passwordForm.new_password !== passwordForm.new_password_confirmation) {
      setPasswordError("كلمات المرور الجديدة غير متطابقة");
      return;
    }

    if (passwordForm.new_password.length < 8) {
      setPasswordError("كلمة المرور يجب أن تكون 8 أحرف على الأقل");
      return;
    }

    try {
      setPasswordLoading(true);
      await authAPI.changePassword(passwordForm);
      setSuccessMessage("تم تغيير كلمة المرور بنجاح");
      setPasswordForm({
        current_password: "",
        new_password: "",
        new_password_confirmation: "",
      });
      setShowPasswordForm(false);
      setTimeout(() => setSuccessMessage(""), 3000);
    } catch (err) {
      setPasswordError(handleAPIError(err));
    } finally {
      setPasswordLoading(false);
    }
  };

  const handleLogout = async () => {
    try {
      await authAPI.logout();
      localStorage.removeItem("auth_token");
      localStorage.removeItem("user_data");
      router.push("/login");
    } catch (err) {
      console.error("Logout error:", err);
      router.push("/login");
    }
  };

  const handleDownloadDocument = async (docId: number, docType: string) => {
    try {
      setDownloadingDocId(docId);
      const response = await authAPI.downloadDocument(docId);

      // Create blob and download
      const url = window.URL.createObjectURL(response.data);
      const link = document.createElement("a");
      link.href = url;
      link.download = `${docType}-${Date.now()}`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error("Download error:", err);
    } finally {
      setDownloadingDocId(null);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "approved":
        return "text-green-600";
      case "rejected":
        return "text-red-600";
      case "pending":
        return "text-yellow-600";
      default:
        return "text-gray-600";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "approved":
        return "موافق عليه";
      case "rejected":
        return "مرفوض";
      case "pending":
        return "قيد الانتظار";
      default:
        return status;
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!profile) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-6 text-center">
        <AlertCircle className="w-12 h-12 text-red-600 mx-auto mb-4" />
        <p className="text-red-800">فشل في تحميل ملف المستخدم</p>
        <button
          onClick={loadProfile}
          className="mt-4 bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg transition"
        >
          أعد المحاولة
        </button>
      </div>
    );
  }

  return (
    <div className="max-w-4xl">
      {/* Success Message */}
      {successMessage && (
        <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-6 flex items-center">
          <CheckCircle className="w-5 h-5 text-green-600 ml-3" />
          <p className="text-green-800">{successMessage}</p>
        </div>
      )}

      {/* Error Message */}
      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6 flex items-center">
          <AlertCircle className="w-5 h-5 text-red-600 ml-3" />
          <p className="text-red-800">{error}</p>
        </div>
      )}

      {/* Profile Header Card */}
      <div className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-lg shadow-lg p-8 text-white mb-6">
        <div className="flex items-center mb-6">
          <div className="w-16 h-16 bg-white bg-opacity-20 rounded-full flex items-center justify-center">
            <User className="w-8 h-8" />
          </div>
          <div className="mr-4">
            <h1 className="text-3xl font-bold">{profile.full_name}</h1>
            <p className="text-blue-100">{profile.email}</p>
          </div>
        </div>

        <div className="grid grid-cols-2 gap-4">
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <p className="text-blue-100 text-sm">الحالة</p>
            <p
              className={`text-lg font-semibold ${getStatusColor(
                profile.status
              )}`}
            >
              {getStatusLabel(profile.status)}
            </p>
          </div>
          <div className="bg-white bg-opacity-10 rounded-lg p-4">
            <p className="text-blue-100 text-sm">الدور</p>
            <p className="text-lg font-semibold capitalize">{profile.role}</p>
          </div>
        </div>
      </div>

      {/* Profile Information Cards */}
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 className="text-xl font-bold text-gray-800 mb-4">معلومات الحساب</h2>

        <div className="space-y-4">
          {/* Email */}
          <div className="flex items-center p-4 bg-gray-50 rounded-lg">
            <Mail className="w-5 h-5 text-blue-600 ml-3" />
            <div>
              <p className="text-gray-600 text-sm">البريد الإلكتروني</p>
              <p className="text-gray-900 font-medium">{profile.email}</p>
            </div>
          </div>

          {/* Status */}
          <div className="flex items-center p-4 bg-gray-50 rounded-lg">
            <CheckCircle className="w-5 h-5 text-blue-600 ml-3" />
            <div>
              <p className="text-gray-600 text-sm">حالة الحساب</p>
              <p className={`font-medium ${getStatusColor(profile.status)}`}>
                {getStatusLabel(profile.status)}
              </p>
            </div>
          </div>

          {/* Active Status */}
          <div className="flex items-center p-4 bg-gray-50 rounded-lg">
            <AlertCircle className="w-5 h-5 text-blue-600 ml-3" />
            <div>
              <p className="text-gray-600 text-sm">حالة النشاط</p>
              <p
                className={`font-medium ${
                  profile.is_active ? "text-green-600" : "text-red-600"
                }`}
              >
                {profile.is_active ? "نشط" : "غير نشط"}
              </p>
            </div>
          </div>

          {/* Categories */}
          {profile.categories && profile.categories.length > 0 && (
            <div className="p-4 bg-gray-50 rounded-lg">
              <p className="text-gray-600 text-sm mb-2">الفئات</p>
              <div className="flex flex-wrap gap-2">
                {profile.categories.map((category, index) => (
                  <span
                    key={index}
                    className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm"
                  >
                    {category}
                  </span>
                ))}
              </div>
            </div>
          )}

          {/* Last Login */}
          {profile.last_login_at && (
            <div className="flex items-center p-4 bg-gray-50 rounded-lg">
              <AlertCircle className="w-5 h-5 text-blue-600 ml-3" />
              <div>
                <p className="text-gray-600 text-sm">آخر تسجيل دخول</p>
                <p className="text-gray-900 font-medium">
                  {new Date(profile.last_login_at).toLocaleString("ar-SA")}
                </p>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Exam Statistics Card */}
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 className="text-xl font-bold text-gray-800 mb-4">
          إحصائيات الاختبار
        </h2>

        <div className="grid grid-cols-3 gap-4">
          <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg p-4 text-center">
            <BarChart3 className="w-8 h-8 text-blue-600 mx-auto mb-2" />
            <p className="text-gray-600 text-sm">الاختبارات المكتملة</p>
            <p className="text-2xl font-bold text-blue-600">
              {profile.has_completed_exam ? "نعم" : "لا"}
            </p>
          </div>

          <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-lg p-4 text-center">
            <BarChart3 className="w-8 h-8 text-purple-600 mx-auto mb-2" />
            <p className="text-gray-600 text-sm">عدد التقييمات</p>
            <p className="text-2xl font-bold text-purple-600">
              {profile.evaluation_count}
            </p>
          </div>

          <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg p-4 text-center">
            <CheckCircle className="w-8 h-8 text-green-600 mx-auto mb-2" />
            <p className="text-gray-600 text-sm">تم التقييم</p>
            <p className="text-2xl font-bold text-green-600">
              {profile.is_evaluated ? "نعم" : "لا"}
            </p>
          </div>
        </div>
      </div>

      {/* Change Password Section */}
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-xl font-bold text-gray-800">الأمان</h2>
          {!showPasswordForm && (
            <button
              onClick={() => setShowPasswordForm(true)}
              className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg transition"
            >
              تغيير كلمة المرور
            </button>
          )}
        </div>

        {showPasswordForm && (
          <form onSubmit={handleChangePassword} className="space-y-4">
            {passwordError && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 flex items-center">
                <AlertCircle className="w-5 h-5 text-red-600 ml-3" />
                <p className="text-red-800">{passwordError}</p>
              </div>
            )}

            <div>
              <label className="block text-gray-700 font-medium mb-2">
                كلمة المرور الحالية
              </label>
              <input
                type="password"
                value={passwordForm.current_password}
                onChange={(e) =>
                  setPasswordForm({
                    ...passwordForm,
                    current_password: e.target.value,
                  })
                }
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="أدخل كلمة المرور الحالية"
                required
              />
            </div>

            <div>
              <label className="block text-gray-700 font-medium mb-2">
                كلمة المرور الجديدة
              </label>
              <input
                type="password"
                value={passwordForm.new_password}
                onChange={(e) =>
                  setPasswordForm({
                    ...passwordForm,
                    new_password: e.target.value,
                  })
                }
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="أدخل كلمة المرور الجديدة"
                required
              />
            </div>

            <div>
              <label className="block text-gray-700 font-medium mb-2">
                تأكيد كلمة المرور الجديدة
              </label>
              <input
                type="password"
                value={passwordForm.new_password_confirmation}
                onChange={(e) =>
                  setPasswordForm({
                    ...passwordForm,
                    new_password_confirmation: e.target.value,
                  })
                }
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="أعد إدخال كلمة المرور الجديدة"
                required
              />
            </div>

            <div className="flex gap-3">
              <button
                type="submit"
                disabled={passwordLoading}
                className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 rounded-lg transition disabled:opacity-50"
              >
                {passwordLoading ? "جاري التحديث..." : "تحديث كلمة المرور"}
              </button>
              <button
                type="button"
                onClick={() => {
                  setShowPasswordForm(false);
                  setPasswordForm({
                    current_password: "",
                    new_password: "",
                    new_password_confirmation: "",
                  });
                  setPasswordError("");
                }}
                className="flex-1 bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 rounded-lg transition"
              >
                إلغاء
              </button>
            </div>
          </form>
        )}
      </div>

      {/* Applicant Data Section */}
      {applicantData && (
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <h2 className="text-2xl font-bold text-gray-800 mb-6">
            بيانات المتقدم
          </h2>

          {/* Personal Information */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-blue-500">
              المعلومات الشخصية
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الاسم (عربي)</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.full_name_ar}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الاسم (English)</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.full_name_en}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">رقم الهوية</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.national_id}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">رقم الجواز</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.passport_number || "غير محدد"}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">تاريخ الميلاد</p>
                <p className="text-gray-900 font-medium">
                  {new Date(applicantData.dob).toLocaleDateString("ar-SA")}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">النوع</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.gender === "male" ? "ذكر" : "أنثى"}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الحالة العسكرية</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.military_status || "غير محدد"}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">مكان الولادة</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.birth_place}
                </p>
              </div>
            </div>
          </div>

          {/* Contact Information */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-green-500">
              معلومات التواصل
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <Phone className="w-5 h-5 text-green-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">الهاتف</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.mobile}
                  </p>
                </div>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <Phone className="w-5 h-5 text-green-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">WhatsApp</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.whatsapp}
                  </p>
                </div>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <Mail className="w-5 h-5 text-green-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">
                    البريد الإلكتروني
                  </p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.email}
                  </p>
                </div>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <MapPin className="w-5 h-5 text-green-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">المحافظة</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.governorate?.name || "غير محدد"}
                  </p>
                </div>
              </div>
              <div className="md:col-span-2 bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">العنوان</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.address}
                </p>
              </div>
              <div className="md:col-span-2 bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">المنطقة</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.district}
                </p>
              </div>
            </div>
          </div>

          {/* Education Information */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-purple-500">
              المعلومات التعليمية
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الجامعة</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.university}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الكلية</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.college}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">التخصص</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.major}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">الدرجة العلمية</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.degree}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">هل تدرس حاليا؟</p>
                <p className="text-gray-900 font-medium">
                  {applicantData.is_studying ? "نعم" : "لا"}
                </p>
              </div>
              {applicantData.is_studying && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">السنة الدراسية</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.study_year}
                  </p>
                </div>
              )}
              {!applicantData.is_studying && applicantData.grad_year && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">سنة التخرج</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.grad_year}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Work Information */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-indigo-500">
              معلومات العمل
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <Briefcase className="w-5 h-5 text-indigo-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">الوظيفة الحالية</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.current_job || "غير محدد"}
                  </p>
                </div>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                <Briefcase className="w-5 h-5 text-indigo-600 ml-3 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-600 text-sm mb-1">جهة التوظيف</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.employer || "غير محدد"}
                  </p>
                </div>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">
                  الخبرة (القطاع العام)
                </p>
                <p className="text-gray-900 font-medium">
                  {applicantData.experience_public || "غير محدد"}
                </p>
              </div>
              <div className="bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">
                  الخبرة (وزارة الخارجية)
                </p>
                <p className="text-gray-900 font-medium">
                  {applicantData.experience_ministry || "غير محدد"}
                </p>
              </div>
              <div className="md:col-span-2 bg-gray-50 p-4 rounded-lg">
                <p className="text-gray-600 text-sm mb-1">
                  الخبرة (منظمات أخرى)
                </p>
                <p className="text-gray-900 font-medium">
                  {applicantData.experience_other_orgs || "غير محدد"}
                </p>
              </div>
            </div>
          </div>

          {/* Skills and Interests */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-red-500">
              المهارات والاهتمامات
            </h3>
            <div className="space-y-4">
              {applicantData.skills_hobbies && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">
                    المهارات والهوايات
                  </p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.skills_hobbies}
                  </p>
                </div>
              )}
              {applicantData.sports && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">الرياضة</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.sports}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Travel and Memberships */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-orange-500">
              السفر والعضويات
            </h3>
            <div className="space-y-4">
              {applicantData.visited_countries && (
                <div className="bg-gray-50 p-4 rounded-lg flex items-start">
                  <Globe className="w-5 h-5 text-orange-600 ml-3 mt-1 flex-shrink-0" />
                  <div>
                    <p className="text-gray-600 text-sm mb-1">الدول المزارة</p>
                    <p className="text-gray-900 font-medium">
                      {applicantData.visited_countries}
                    </p>
                  </div>
                </div>
              )}
              {applicantData.center_memberships && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">عضويات المراكز</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.center_memberships}
                  </p>
                </div>
              )}
              {applicantData.club_memberships && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">عضويات الأندية</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.club_memberships}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Personal Assessment */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-cyan-500">
              التقييم الشخصي
            </h3>
            <div className="space-y-4">
              {applicantData.motivation && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">الدوافع</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.motivation}
                  </p>
                </div>
              )}
              {applicantData.strengths && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">المميزات</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.strengths}
                  </p>
                </div>
              )}
              {applicantData.weaknesses && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">نقاط الضعف</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.weaknesses}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Physical Information */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-pink-500">
              المعلومات البدنية
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {applicantData.height && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">الطول</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.height}
                  </p>
                </div>
              )}
              {applicantData.weight && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">الوزن</p>
                  <p className="text-gray-900 font-medium">
                    {applicantData.weight}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Social Media */}
          <div className="mb-8">
            <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-blue-500">
              وسائل التواصل الاجتماعي
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {applicantData.facebook && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">Facebook</p>
                  <p className="text-gray-900 font-medium text-sm break-all">
                    {applicantData.facebook}
                  </p>
                </div>
              )}
              {applicantData.instagram && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">Instagram</p>
                  <p className="text-gray-900 font-medium text-sm break-all">
                    {applicantData.instagram}
                  </p>
                </div>
              )}
              {applicantData.linkedin && (
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-600 text-sm mb-1">LinkedIn</p>
                  <p className="text-gray-900 font-medium text-sm break-all">
                    {applicantData.linkedin}
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Nationalities Section */}
          {applicantData.nationalities &&
            applicantData.nationalities.length > 0 && (
              <div className="mb-8">
                <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-teal-500">
                  الجنسيات
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {applicantData.nationalities.map((nationality, index) => (
                    <div
                      key={nationality.id}
                      className="bg-gray-50 p-4 rounded-lg"
                    >
                      <p className="text-gray-600 text-sm mb-1">
                        الجنسية {index + 1}
                      </p>
                      <p className="text-gray-900 font-medium">
                        {nationality.country?.name_ar || `جنسية #${index + 1}`}
                      </p>
                    </div>
                  ))}
                </div>
              </div>
            )}

          {/* Languages Section */}
          {applicantData.languages && applicantData.languages.length > 0 && (
            <div className="mb-8">
              <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-sky-500">
                اللغات
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {applicantData.languages.map((language, index) => (
                  <div key={language.id} className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-gray-600 text-sm mb-1">
                      اللغة {index + 1}
                    </p>
                    <p className="text-gray-900 font-medium">
                      {language.language?.name || `لغة #${index + 1}`}
                    </p>
                    {language.proficiency && (
                      <p className="text-gray-600 text-xs mt-1">
                        المستوى: {language.proficiency}
                      </p>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Degrees Section */}
          {applicantData.degrees && applicantData.degrees.length > 0 && (
            <div className="mb-8">
              <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-lime-500">
                الشهادات الأخرى
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {applicantData.degrees.map((degree, index) => (
                  <div key={degree.id} className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-gray-600 text-sm mb-1">
                      الشهادة {index + 1}
                    </p>
                    <p className="text-gray-900 font-medium">
                      {degree.title || "شهادة"}
                    </p>
                    {degree.issuing_body && (
                      <p className="text-gray-600 text-sm mt-1">
                        {degree.issuing_body}
                      </p>
                    )}
                    {degree.issue_date && (
                      <p className="text-gray-600 text-xs mt-1">
                        {new Date(degree.issue_date).toLocaleDateString(
                          "ar-SA"
                        )}
                      </p>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Documents Section */}
          {applicantData.documents && applicantData.documents.length > 0 && (
            <div className="mb-8">
              <h3 className="text-lg font-semibold text-gray-700 mb-4 pb-2 border-b-2 border-amber-500">
                المستندات
              </h3>
              <div className="space-y-3">
                {applicantData.documents.map((doc) => (
                  <div
                    key={doc.id}
                    className="bg-gray-50 p-4 rounded-lg flex items-center justify-between hover:bg-gray-100 transition"
                  >
                    <div className="flex items-center">
                      {doc.type === "cv" && (
                        <FileText className="w-6 h-6 text-blue-600 ml-3" />
                      )}
                      {doc.type.includes("photo") && (
                        <ImageIcon className="w-6 h-6 text-purple-600 ml-3" />
                      )}
                      {!doc.type.includes("photo") && doc.type !== "cv" && (
                        <FileText className="w-6 h-6 text-gray-600 ml-3" />
                      )}
                      <div>
                        <p className="text-gray-900 font-medium">
                          {getDocumentLabel(doc.type)}
                        </p>
                        <p className="text-gray-600 text-xs mt-1">
                          {new Date(doc.created_at).toLocaleDateString("ar-SA")}
                        </p>
                      </div>
                    </div>
                    <button
                      onClick={() => handleDownloadDocument(doc.id, doc.type)}
                      disabled={downloadingDocId === doc.id}
                      className="bg-blue-600 hover:bg-blue-700 text-white p-2 rounded-lg transition flex items-center gap-1 disabled:opacity-50"
                      title="تحميل الملف"
                    >
                      <Download
                        className={`w-4 h-4 ${
                          downloadingDocId === doc.id ? "animate-bounce" : ""
                        }`}
                      />
                    </button>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      )}

      {/* Logout Button */}
      <button
        onClick={handleLogout}
        className="w-full bg-red-600 hover:bg-red-700 text-white py-3 rounded-lg transition flex items-center justify-center"
      >
        <LogOut className="w-5 h-5 ml-2" />
        تسجيل الخروج
      </button>
    </div>
  );
}
