"use client";

import { useEffect, useState } from "react";
import { userAPI, authAPI, handleAPIError } from "@/lib/api";
import { getUser } from "@/lib/auth";
import {
  FileText,
  Clock,
  CheckCircle,
  AlertCircle,
  BarChart3,
  User,
} from "lucide-react";
import Link from "next/link";

export default function UserDashboard() {
  const [profile, setProfile] = useState<any>(null);
  const [exams, setExams] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const user = getUser();

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [profileResponse, examsResponse] = await Promise.all([
        authAPI.getProfile(),
        userAPI.getExams(),
      ]);
      setProfile(profileResponse.data.data);
      setExams(examsResponse.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  const getStatusInfo = (status: string) => {
    const statusMap: Record<
      string,
      { label: string; color: string; icon: any }
    > = {
      pending: {
        label: "في انتظار الاختبار",
        color: "yellow",
        icon: Clock,
      },
      exam_completed: {
        label: "أكملت الاختبار",
        color: "blue",
        icon: CheckCircle,
      },
      under_evaluation: {
        label: "تحت التقييم",
        color: "purple",
        icon: BarChart3,
      },
      evaluated: {
        label: "تم التقييم",
        color: "indigo",
        icon: CheckCircle,
      },
      accepted: {
        label: "تم القبول",
        color: "green",
        icon: CheckCircle,
      },
      rejected: {
        label: "تم الرفض",
        color: "red",
        icon: AlertCircle,
      },
    };
    return statusMap[status] || statusMap.pending;
  };

  const statusInfo = getStatusInfo(profile.status);
  const StatusIcon = statusInfo.icon;

  return (
    <div className="space-y-6" dir="rtl">
      {/* Welcome Section */}
      <div className="bg-gradient-to-br from-blue-600 to-indigo-600 rounded-xl shadow-lg p-8 text-white">
        <div className="flex items-center gap-4 mb-4">
          <div className="w-16 h-16 bg-white bg-opacity-20 rounded-full flex items-center justify-center">
            <User className="w-8 h-8" />
          </div>
          <div>
            <h1 className="text-3xl font-bold">مرحباً، {profile.name}</h1>
            <p className="text-blue-100">برنامج الدبلوماسية الشبابية</p>
          </div>
        </div>

        <div className="flex items-center gap-3 p-4 bg-white bg-opacity-20 rounded-lg">
          <StatusIcon className="w-6 h-6" />
          <div>
            <p className="text-sm text-blue-100">حالة التقديم</p>
            <p className="text-lg font-semibold">{statusInfo.label}</p>
          </div>
        </div>
      </div>

      {/* Status Card */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">حالة طلبك</h3>

        <div className="space-y-4">
          {/* Step 1: Registration */}
          <div className="flex items-start gap-4">
            <div className="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center flex-shrink-0">
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">التسجيل</h4>
              <p className="text-sm text-gray-600">تم التسجيل بنجاح</p>
            </div>
          </div>

          {/* Step 2: Exam */}
          <div className="flex items-start gap-4">
            <div
              className={`w-10 h-10 rounded-full flex items-center justify-center flex-shrink-0 ${
                [
                  "exam_completed",
                  "under_evaluation",
                  "evaluated",
                  "accepted",
                  "rejected",
                ].includes(profile.status)
                  ? "bg-green-100"
                  : "bg-gray-100"
              }`}
            >
              {[
                "exam_completed",
                "under_evaluation",
                "evaluated",
                "accepted",
                "rejected",
              ].includes(profile.status) ? (
                <CheckCircle className="w-6 h-6 text-green-600" />
              ) : (
                <Clock className="w-6 h-6 text-gray-400" />
              )}
            </div>
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">الاختبار</h4>
              {profile.status === "pending" ? (
                <p className="text-sm text-gray-600">في انتظار نشر الاختبار</p>
              ) : (
                <p className="text-sm text-gray-600">تم إكمال الاختبار</p>
              )}
            </div>
          </div>

          {/* Step 3: Evaluation */}
          <div className="flex items-start gap-4">
            <div
              className={`w-10 h-10 rounded-full flex items-center justify-center flex-shrink-0 ${
                ["evaluated", "accepted", "rejected"].includes(profile.status)
                  ? "bg-green-100"
                  : profile.status === "under_evaluation"
                  ? "bg-yellow-100"
                  : "bg-gray-100"
              }`}
            >
              {["evaluated", "accepted", "rejected"].includes(
                profile.status
              ) ? (
                <CheckCircle className="w-6 h-6 text-green-600" />
              ) : profile.status === "under_evaluation" ? (
                <Clock className="w-6 h-6 text-yellow-600" />
              ) : (
                <Clock className="w-6 h-6 text-gray-400" />
              )}
            </div>
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">التقييم</h4>
              {profile.status === "under_evaluation" && (
                <p className="text-sm text-gray-600">
                  جاري تقييم طلبك من قبل اللجنة
                </p>
              )}
              {["evaluated", "accepted", "rejected"].includes(
                profile.status
              ) && <p className="text-sm text-gray-600">تم التقييم</p>}
              {![
                "under_evaluation",
                "evaluated",
                "accepted",
                "rejected",
              ].includes(profile.status) && (
                <p className="text-sm text-gray-600">في انتظار التقييم</p>
              )}
            </div>
          </div>

          {/* Step 4: Final Decision */}
          <div className="flex items-start gap-4">
            <div
              className={`w-10 h-10 rounded-full flex items-center justify-center flex-shrink-0 ${
                profile.status === "accepted"
                  ? "bg-green-100"
                  : profile.status === "rejected"
                  ? "bg-red-100"
                  : "bg-gray-100"
              }`}
            >
              {profile.status === "accepted" ? (
                <CheckCircle className="w-6 h-6 text-green-600" />
              ) : profile.status === "rejected" ? (
                <AlertCircle className="w-6 h-6 text-red-600" />
              ) : (
                <Clock className="w-6 h-6 text-gray-400" />
              )}
            </div>
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">القرار النهائي</h4>
              {profile.status === "accepted" && (
                <p className="text-sm text-green-600 font-medium">
                  مبروك! تم قبولك في البرنامج
                </p>
              )}
              {profile.status === "rejected" && (
                <p className="text-sm text-red-600">
                  نأسف، لم يتم قبولك في هذه المرة
                </p>
              )}
              {!["accepted", "rejected"].includes(profile.status) && (
                <p className="text-sm text-gray-600">
                  في انتظار القرار النهائي
                </p>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Available Exams */}
      {profile.status === "pending" && exams.length > 0 && (
        <div className="bg-white rounded-xl shadow-lg p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
            <FileText className="w-6 h-6 text-blue-600" />
            الاختبارات المتاحة
          </h3>
          <div className="space-y-3">
            {exams.map((exam) => (
              <div
                key={exam.id}
                className="flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition-colors"
              >
                <div>
                  <h4 className="font-medium text-gray-900">{exam.title}</h4>
                  <p className="text-sm text-gray-600">{exam.description}</p>
                  <p className="text-sm text-gray-500 mt-1">
                    المدة: {exam.duration_minutes} دقيقة
                  </p>
                </div>
                <Link
                  href={`/dashboard/exam/${exam.id}`}
                  className="px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-medium transition-colors"
                >
                  بدء الاختبار
                </Link>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* No Exams Available */}
      {profile.status === "pending" && exams.length === 0 && (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-500">لا توجد اختبارات متاحة حالياً</p>
          <p className="text-sm text-gray-400 mt-2">
            سيتم إشعارك عند نشر الاختبار
          </p>
        </div>
      )}
    </div>
  );
}
