"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import {
  Search,
  Filter,
  Eye,
  CheckCircle,
  XCircle,
  Clock,
  FileText,
  ChevronLeft,
  ChevronRight,
  Users as UsersIcon,
  Mail,
  Calendar,
  Tag,
  Download,
  RefreshCw,
} from "lucide-react";
import Link from "next/link";
import AssignCategoriesModal from "@/components/admin/AssignCategoriesModal";

interface User {
  id: number;
  name: string;
  full_name: string;
  email: string;
  status: string;
  created_at: string;
  exam_attempts_count?: number;
  evaluations_count?: number;
  categories: Array<{ id: number; name: string }>;
}

interface Category {
  id: number;
  name: string;
  description?: string;
}

export default function UsersPage() {
  const [users, setUsers] = useState<User[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState("");
  const [categoryFilter, setCategoryFilter] = useState("");
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalUsers, setTotalUsers] = useState(0);
  const [selectedUsers, setSelectedUsers] = useState<Set<number>>(new Set());
  const [assignModalOpen, setAssignModalOpen] = useState(false);
  const [modalUser, setModalUser] = useState<User | null>(null);
  const [exporting, setExporting] = useState(false);
  const [formalPhotos, setFormalPhotos] = useState<Record<number, string>>({});
  const [notification, setNotification] = useState<{
    show: boolean;
    message: string;
    type: "success" | "error";
  }>({ show: false, message: "", type: "success" });

  useEffect(() => {
    loadCategories();
    loadUsers();
  }, []);

  useEffect(() => {
    loadUsers();
  }, [search, statusFilter, categoryFilter, currentPage]);

  const showNotification = (message: string, type: "success" | "error") => {
    setNotification({ show: true, message, type });
    setTimeout(() => {
      setNotification({ show: false, message: "", type: "success" });
    }, 3000);
  };

  const loadCategories = async () => {
    try {
      const response = await adminAPI.getCategories();
      // Handle both response formats
      const categoriesData = Array.isArray(response.data.data)
        ? response.data.data
        : Array.isArray(response.data)
        ? response.data
        : [];
      setCategories(categoriesData);
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    }
  };

  const loadUsers = async () => {
    setLoading(true);
    try {
      const response = await adminAPI.getUsers({
        search,
        status: statusFilter || undefined,
        category_id: categoryFilter ? parseInt(categoryFilter) : undefined,
        page: currentPage,
      });
      const usersData = response.data.data.data.map((user: any) => ({
        id: user.id,
        email: user.email,
        full_name: user.full_name,
        status: user.status,
        created_at: user.created_at,
        role: user.role,
        is_active: user.is_active,
        name: user.full_name || "",
        exam_attempts_count: user.exam_attempts_count || 0,
        evaluations_count: user.evaluations_count || 0,
        categories: Array.isArray(user.categories)
          ? user.categories.map((cat: any) => ({
              id: cat.id,
              name: cat.name,
              description: cat.description,
            }))
          : [],
      }));
      setUsers(usersData);
      setTotalPages(response.data.data.last_page);
      setTotalUsers(response.data.data.total);

      // Load formal photos for all users
      const photoMap: Record<number, string> = {};
      for (const user of usersData) {
        try {
          const photoResponse = await fetch(
            `${process.env.NEXT_PUBLIC_API_URL}/admin/users/${user.id}/documents/photo_official/preview`,
            {
              headers: {
                Authorization: `Bearer ${localStorage.getItem("auth_token")}`,
              },
            }
          );

          if (photoResponse.ok) {
            const blob = await photoResponse.blob();
            const objectUrl = URL.createObjectURL(blob);
            photoMap[user.id] = objectUrl;
          }
        } catch (err) {
          console.error(`Failed to load photo for user ${user.id}`);
        }
      }

      setFormalPhotos(photoMap);
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    } finally {
      setLoading(false);
    }
  };

  const toggleUserSelection = (userId: number) => {
    const newSelected = new Set(selectedUsers);
    if (newSelected.has(userId)) {
      newSelected.delete(userId);
    } else {
      newSelected.add(userId);
    }
    setSelectedUsers(newSelected);
  };

  const toggleSelectAll = () => {
    if (selectedUsers.size === users.length) {
      setSelectedUsers(new Set());
    } else {
      setSelectedUsers(new Set(users.map((u) => u.id)));
    }
  };

  const openAssignModal = (user: User) => {
    setModalUser(user);
    setAssignModalOpen(true);
  };

  const exportToExcel = async () => {
    setExporting(true);
    try {
      // Fetch all users with current filters (without pagination)
      const response = await adminAPI.exportUsers({
        search,
        status: statusFilter || undefined,
        category_id: categoryFilter ? parseInt(categoryFilter) : undefined,
      });

      // Create a Blob from the response
      const blob = new Blob([response.data], {
        type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
      });

      // Create a download link
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement("a");
      link.href = url;
      link.download = `applicants-${
        new Date().toISOString().split("T")[0]
      }.xlsx`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);

      showNotification("تم تحميل الملف بنجاح", "success");
    } catch (err: any) {
      showNotification(handleAPIError(err) || "فشل تحميل الملف", "error");
    } finally {
      setExporting(false);
    }
  };

  const getStatusBadge = (status: string) => {
    const statusConfig: Record<
      string,
      { label: string; className: string; icon: any }
    > = {
      pending: {
        label: "قيد الانتظار",
        className: "bg-yellow-50 text-yellow-700 border border-yellow-200",
        icon: Clock,
      },
      exam_completed: {
        label: "أكمل الاختبار",
        className: "bg-blue-50 text-blue-700 border border-blue-200",
        icon: CheckCircle,
      },
      under_evaluation: {
        label: "تحت التقييم",
        className: "bg-purple-50 text-purple-700 border border-purple-200",
        icon: FileText,
      },
      evaluated: {
        label: "تم التقييم",
        className: "bg-indigo-50 text-indigo-700 border border-indigo-200",
        icon: CheckCircle,
      },
      accepted: {
        label: "مقبول",
        className: "bg-green-50 text-green-700 border border-green-200",
        icon: CheckCircle,
      },
      rejected: {
        label: "مرفوض",
        className: "bg-red-50 text-red-700 border border-red-200",
        icon: XCircle,
      },
    };

    const config = statusConfig[status] || {
      label: status,
      className: "bg-gray-50 text-gray-700 border border-gray-200",
      icon: Clock,
    };
    const Icon = config.icon;
    return (
      <span
        className={`inline-flex items-center gap-1.5 px-3 py-1.5 rounded-lg text-xs font-semibold ${config.className}`}
      >
        <Icon className="w-3.5 h-3.5" />
        {config.label}
      </span>
    );
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("ar-EG", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const getStatusStats = () => {
    const stats = {
      total: totalUsers,
      pending: 0,
      accepted: 0,
      rejected: 0,
      under_evaluation: 0,
      exam_completed: 0,
      evaluated: 0,
    };
    users.forEach((user) => {
      if (user.status === "pending") stats.pending++;
      if (user.status === "accepted") stats.accepted++;
      if (user.status === "rejected") stats.rejected++;
      if (user.status === "evaluated") stats.evaluated++;
      if (user.status === "under_evaluation") stats.under_evaluation++;
      if (user.status === "exam_completed") stats.exam_completed++;
    });
    return stats;
  };

  const stats = getStatusStats();

  return (
    <div
      className="min-h-screen bg-gradient-to-br from-gray-50 to-blue-50 p-6"
      dir="rtl"
    >
      {/* Notification Toast */}
      {notification.show && (
        <div className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 animate-slide-down">
          <div
            className={`${
              notification.type === "success" ? "bg-green-500" : "bg-red-500"
            } text-white px-6 py-3 rounded-lg shadow-lg flex items-center gap-3`}
          >
            <span>{notification.message}</span>
          </div>
        </div>
      )}

      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <UsersIcon className="w-8 h-8 text-blue-600" />
                </div>
                إدارة المستخدمين
              </h1>
              <p className="text-gray-600 mt-2 mr-14">
                عرض وإدارة جميع المتقدمين والمستخدمين
              </p>
            </div>
            <button
              onClick={loadUsers}
              className="flex items-center gap-2 bg-blue-50 hover:bg-blue-100 text-blue-600 px-4 py-2.5 rounded-xl transition-all duration-200"
            >
              <RefreshCw className="w-4 h-4" />
              تحديث
            </button>
          </div>
        </div>

        {/* Statistics Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">
                  إجمالي المستخدمين
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {stats.total}
                </p>
              </div>
              <div className="p-3 bg-blue-50 rounded-xl">
                <UsersIcon className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">
                  قيد الانتظار
                </p>
                <p className="text-3xl font-bold text-yellow-600">
                  {stats.pending}
                </p>
              </div>
              <div className="p-3 bg-yellow-50 rounded-xl">
                <Clock className="w-6 h-6 text-yellow-600" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">مقبول</p>
                <p className="text-3xl font-bold text-green-600">
                  {stats.accepted}
                </p>
              </div>
              <div className="p-3 bg-green-50 rounded-xl">
                <CheckCircle className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">مرفوض</p>
                <p className="text-3xl font-bold text-red-600">
                  {stats.rejected}
                </p>
              </div>
              <div className="p-3 bg-red-50 rounded-xl">
                <XCircle className="w-6 h-6 text-red-600" />
              </div>
            </div>
          </div>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">
                  المستخدمون المُقيمون
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {stats.evaluated}
                </p>
              </div>
              <div className="p-3 bg-blue-50 rounded-xl">
                <UsersIcon className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">
                  أكملوا الإختبار
                </p>
                <p className="text-3xl font-bold text-yellow-600">
                  {stats.exam_completed}
                </p>
              </div>
              <div className="p-3 bg-yellow-50 rounded-xl">
                <Clock className="w-6 h-6 text-yellow-600" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 font-medium mb-1">
                  تحت التقييم
                </p>
                <p className="text-3xl font-bold text-green-600">
                  {stats.under_evaluation}
                </p>
              </div>
              <div className="p-3 bg-green-50 rounded-xl">
                <CheckCircle className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </div>
        </div>

        {/* Bulk selection info */}
        {selectedUsers.size > 0 && (
          <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-2xl p-4 flex items-center justify-between shadow-sm">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-blue-100 rounded-lg">
                <CheckCircle className="w-5 h-5 text-blue-600" />
              </div>
              <span className="text-blue-900 font-semibold">
                تم تحديد {selectedUsers.size} مستخدم
              </span>
            </div>
            <button
              onClick={() => setSelectedUsers(new Set())}
              className="text-sm text-blue-600 hover:text-blue-800 font-medium px-4 py-2 hover:bg-blue-100 rounded-lg transition-colors"
            >
              إلغاء التحديد
            </button>
          </div>
        )}

        {/* Filters */}
        <div className="bg-white rounded-2xl shadow-sm p-6 border border-gray-100">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {/* Search */}
            <div className="relative">
              <Search className="absolute right-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                value={search}
                onChange={(e) => {
                  setSearch(e.target.value);
                  setCurrentPage(1);
                }}
                placeholder="البحث بالاسم أو البريد الإلكتروني..."
                className="w-full pr-12 pl-4 py-3.5 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
              />
            </div>

            {/* Status Filter */}
            <div className="relative">
              <Filter className="absolute right-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <select
                value={statusFilter}
                onChange={(e) => {
                  setStatusFilter(e.target.value);
                  setCurrentPage(1);
                }}
                className="w-full pr-12 pl-4 py-3.5 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all appearance-none cursor-pointer"
              >
                <option value="">جميع الحالات</option>
                <option value="pending">قيد الانتظار</option>
                <option value="exam_completed">أكمل الاختبار</option>
                <option value="under_evaluation">تحت التقييم</option>
                <option value="evaluated">تم التقييم</option>
                <option value="accepted">مقبول</option>
                <option value="rejected">مرفوض</option>
              </select>
            </div>

            {/* Category Filter */}
            <div className="relative">
              <Tag className="absolute right-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
              <select
                value={categoryFilter}
                onChange={(e) => {
                  setCategoryFilter(e.target.value);
                  setCurrentPage(1);
                }}
                className="w-full pr-12 pl-4 py-3.5 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all appearance-none cursor-pointer"
              >
                <option value="">جميع التصنيفات</option>
                {categories.map((cat) => (
                  <option key={cat.id} value={cat.id.toString()}>
                    {cat.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Export Button */}
          <div className="mt-4 flex justify-between items-center">
            <button
              onClick={exportToExcel}
              disabled={exporting || users.length === 0}
              className="flex items-center gap-2 bg-green-50 hover:bg-green-100 disabled:bg-gray-100 text-green-600 disabled:text-gray-400 px-5 py-2.5 rounded-xl transition-all duration-200 border border-green-200 disabled:border-gray-200 font-medium"
            >
              <Download className="w-4 h-4" />
              {exporting ? "جاري التحميل..." : "تحميل Excel"}
            </button>
          </div>
        </div>

        {/* Users Table */}
        <div className="bg-white rounded-2xl shadow-sm overflow-hidden border border-gray-100">
          {loading ? (
            <div className="flex flex-col items-center justify-center p-8 md:p-16">
              <div className="w-16 h-16 border-4 border-blue-600 border-t-transparent rounded-full animate-spin mb-4" />
              <p className="text-gray-600 font-medium">
                جاري تحميل المستخدمين...
              </p>
            </div>
          ) : users.length === 0 ? (
            <div className="text-center p-8 md:p-16">
              <div className="p-4 bg-gray-50 rounded-full w-20 h-20 mx-auto mb-4 flex items-center justify-center">
                <UsersIcon className="w-10 h-10 text-gray-400" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">
                لا توجد نتائج
              </h3>
              <p className="text-gray-500">
                {search || statusFilter
                  ? "جرب تغيير معايير البحث"
                  : "لا يوجد مستخدمين بعد"}
              </p>
            </div>
          ) : (
            <>
              {/* Desktop Table View */}
              <div className="hidden md:block overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gradient-to-r from-gray-50 to-gray-100 border-b border-gray-200">
                    <tr>
                      <th className="px-4 lg:px-6 py-4 text-right">
                        <input
                          type="checkbox"
                          checked={
                            users.length > 0 &&
                            selectedUsers.size === users.length
                          }
                          onChange={toggleSelectAll}
                          className="w-5 h-5 border-gray-300 rounded cursor-pointer text-blue-600 focus:ring-blue-500 focus:ring-2"
                        />
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900">
                        المستخدم
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900 hidden lg:table-cell">
                        الحالة
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900 hidden xl:table-cell">
                        تاريخ التقديم
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900 hidden xl:table-cell">
                        الاختبارات
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900 hidden xl:table-cell">
                        التقييمات
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-right text-sm font-bold text-gray-900 hidden lg:table-cell">
                        التصنيفات
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-center text-sm font-bold text-gray-900">
                        إدارة التصنيفات
                      </th>
                      <th className="px-4 lg:px-6 py-4 text-center text-sm font-bold text-gray-900">
                        عرض التفاصيل
                      </th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-100">
                    {users.map((user) => (
                      <tr
                        key={user.id}
                        className="hover:bg-gradient-to-r hover:from-blue-50/30 hover:to-indigo-50/30 transition-all duration-200"
                      >
                        <td className="px-4 lg:px-6 py-4">
                          <input
                            type="checkbox"
                            checked={selectedUsers.has(user.id)}
                            onChange={() => toggleUserSelection(user.id)}
                            className="w-5 h-5 border-gray-300 rounded cursor-pointer text-blue-600 focus:ring-blue-500 focus:ring-2"
                          />
                        </td>
                        <td className="px-4 lg:px-6 py-4">
                          <div className="flex items-center gap-2 lg:gap-3">
                            <div className="w-10 h-10 lg:w-12 lg:h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-full flex items-center justify-center overflow-hidden flex-shrink-0 shadow-md">
                              {formalPhotos[user.id] ? (
                                <img
                                  src={formalPhotos[user.id]}
                                  alt={user.full_name}
                                  className="w-full h-full object-cover"
                                />
                              ) : (
                                <UsersIcon className="w-5 lg:w-6 h-5 lg:h-6 text-white" />
                              )}
                            </div>
                            <div className="min-w-0 flex-1">
                              <div className="font-semibold text-gray-900 truncate text-sm lg:text-base">
                                {user.full_name}
                              </div>
                              <div className="text-xs lg:text-sm text-gray-500 truncate">
                                {user.email}
                              </div>
                            </div>
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4 hidden lg:table-cell">
                          {getStatusBadge(user.status)}
                        </td>
                        <td className="px-4 lg:px-6 py-4 hidden xl:table-cell">
                          <div className="flex items-center gap-2 text-xs lg:text-sm text-gray-600">
                            <Calendar className="w-4 h-4 flex-shrink-0" />
                            {user.created_at
                              ? formatDate(user.created_at)
                              : "—"}
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4 hidden xl:table-cell">
                          <div className="flex items-center gap-2">
                            <div className="p-2 bg-blue-50 rounded-lg">
                              <FileText className="w-4 h-4 text-blue-600" />
                            </div>
                            <span className="font-semibold text-gray-900">
                              {user.exam_attempts_count}
                            </span>
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4 hidden xl:table-cell">
                          <div className="flex items-center gap-2">
                            <div className="p-2 bg-purple-50 rounded-lg">
                              <CheckCircle className="w-4 h-4 text-purple-600" />
                            </div>
                            <span className="font-semibold text-gray-900">
                              {user.evaluations_count}/5
                            </span>
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4 hidden lg:table-cell">
                          <div className="flex flex-wrap gap-1">
                            {user.categories && user.categories.length > 0 ? (
                              user.categories.slice(0, 2).map((cat) => (
                                <span
                                  key={cat.id}
                                  className="inline-flex items-center gap-1 px-2 py-1 bg-gradient-to-r from-blue-50 to-indigo-50 text-blue-700 text-xs font-medium rounded-lg border border-blue-200"
                                  title={cat.name}
                                >
                                  <Tag className="w-3 h-3 flex-shrink-0" />
                                  <span className="truncate">
                                    {cat.name.substring(0, 15)}
                                  </span>
                                </span>
                              ))
                            ) : (
                              <span className="text-xs text-gray-400">
                                لا يوجد
                              </span>
                            )}
                            {user.categories && user.categories.length > 2 && (
                              <span className="text-xs text-gray-500 px-2 py-1">
                                +{user.categories.length - 2}
                              </span>
                            )}
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4">
                          <div className="flex justify-center">
                            <button
                              onClick={() => openAssignModal(user)}
                              className="inline-flex items-center gap-2 px-3 lg:px-4 py-2 lg:py-2.5 bg-blue-50 hover:bg-blue-100 text-blue-600 hover:text-blue-700 rounded-xl transition-all duration-200 border border-blue-200 hover:border-blue-300 flex-shrink-0"
                              title="تعيين التصنيفات"
                            >
                              <Tag className="w-4 h-4" />
                              <span className="text-xs lg:text-sm font-medium hidden sm:inline">
                                تعيين
                              </span>
                            </button>
                          </div>
                        </td>
                        <td className="px-4 lg:px-6 py-4">
                          <div className="flex justify-center">
                            <Link
                              href={`/admin/users/${user.id}`}
                              className="inline-flex items-center gap-2 px-3 lg:px-4 py-2 lg:py-2.5 bg-gray-50 hover:bg-gray-100 text-gray-700 rounded-xl transition-all duration-200 border border-gray-200 hover:border-gray-300 flex-shrink-0"
                              title="عرض التفاصيل"
                            >
                              <Eye className="w-4 h-4" />
                              <span className="text-xs lg:text-sm font-medium hidden sm:inline">
                                عرض
                              </span>
                            </Link>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {/* Mobile Card View */}
              <div className="md:hidden divide-y divide-gray-200">
                {users.map((user) => (
                  <div
                    key={user.id}
                    className="p-4 hover:bg-gray-50 transition-colors"
                  >
                    {/* Mobile Header */}
                    <div className="flex items-start justify-between gap-2 mb-3">
                      <div className="flex items-start gap-3 flex-1 min-w-0">
                        <input
                          type="checkbox"
                          checked={selectedUsers.has(user.id)}
                          onChange={() => toggleUserSelection(user.id)}
                          className="w-5 h-5 border-gray-300 rounded cursor-pointer text-blue-600 focus:ring-blue-500 focus:ring-2 mt-1 flex-shrink-0"
                        />
                        <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-full flex items-center justify-center overflow-hidden flex-shrink-0 shadow-md">
                          {formalPhotos[user.id] ? (
                            <img
                              src={formalPhotos[user.id]}
                              alt={user.full_name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <UsersIcon className="w-6 h-6 text-white" />
                          )}
                        </div>
                        <div className="min-w-0 flex-1">
                          <div className="font-semibold text-gray-900 text-sm">
                            {user.full_name}
                          </div>
                          <div className="text-xs text-gray-500 truncate">
                            {user.email}
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2 flex-shrink-0">
                        <button
                          onClick={() => openAssignModal(user)}
                          className="p-2 hover:bg-blue-50 text-blue-600 rounded-lg transition-colors"
                          title="تعيين التصنيفات"
                        >
                          <Tag className="w-5 h-5" />
                        </button>
                        <Link
                          href={`/admin/users/${user.id}`}
                          className="p-2 hover:bg-gray-100 text-gray-700 rounded-lg transition-colors"
                          title="عرض التفاصيل"
                        >
                          <Eye className="w-5 h-5" />
                        </Link>
                      </div>
                    </div>

                    {/* Mobile Info Grid */}
                    <div className="space-y-2 text-xs">
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">الحالة:</span>
                        {getStatusBadge(user.status)}
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">تاريخ التقديم:</span>
                        <span className="font-semibold text-gray-900">
                          {user.created_at ? formatDate(user.created_at) : "—"}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">الاختبارات:</span>
                        <span className="font-semibold text-gray-900">
                          {user.exam_attempts_count}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">التقييمات:</span>
                        <span className="font-semibold text-gray-900">
                          {user.evaluations_count}/5
                        </span>
                      </div>
                      {user.categories && user.categories.length > 0 && (
                        <div>
                          <span className="text-gray-600">التصنيفات:</span>
                          <div className="flex flex-wrap gap-1 mt-1">
                            {user.categories.map((cat) => (
                              <span
                                key={cat.id}
                                className="inline-flex items-center gap-1 px-2 py-1 bg-blue-50 text-blue-700 text-xs font-medium rounded-lg border border-blue-200"
                              >
                                <Tag className="w-3 h-3" />
                                {cat.name}
                              </span>
                            ))}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                ))}
              </div>

              {/* Pagination */}
              {totalPages > 1 && (
                <div className="flex items-center justify-between px-6 py-4 border-t border-gray-200 bg-gray-50">
                  <button
                    onClick={() =>
                      setCurrentPage((prev) => Math.max(1, prev - 1))
                    }
                    disabled={currentPage === 1}
                    className="flex items-center gap-2 px-5 py-2.5 border border-gray-300 rounded-xl hover:bg-white disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200 font-medium text-gray-700 disabled:hover:bg-transparent"
                  >
                    <ChevronRight className="w-4 h-4" />
                    السابق
                  </button>
                  <div className="flex items-center gap-2">
                    <span className="text-sm text-gray-600 font-medium">
                      صفحة {currentPage} من {totalPages}
                    </span>
                    <span className="text-xs text-gray-500 bg-gray-100 px-3 py-1 rounded-lg">
                      {totalUsers} مستخدم
                    </span>
                  </div>
                  <button
                    onClick={() =>
                      setCurrentPage((prev) => Math.min(totalPages, prev + 1))
                    }
                    disabled={currentPage === totalPages}
                    className="flex items-center gap-2 px-5 py-2.5 border border-gray-300 rounded-xl hover:bg-white disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200 font-medium text-gray-700 disabled:hover:bg-transparent"
                  >
                    التالي
                    <ChevronLeft className="w-4 h-4" />
                  </button>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      {/* Assign Categories Modal */}
      {modalUser && (
        <AssignCategoriesModal
          isOpen={assignModalOpen}
          onClose={() => {
            setAssignModalOpen(false);
            setModalUser(null);
          }}
          onSuccess={() => {
            setAssignModalOpen(false);
            setModalUser(null);
            loadUsers();
            showNotification("تم تعيين التصنيفات بنجاح", "success");
          }}
          userId={modalUser.id}
          userName={modalUser.full_name}
          currentCategories={modalUser.categories}
        />
      )}

      <style jsx global>{`
        @keyframes slide-down {
          from {
            opacity: 0;
            transform: translate(-50%, -100%);
          }
          to {
            opacity: 1;
            transform: translate(-50%, 0);
          }
        }

        @keyframes slide-up {
          from {
            opacity: 0;
            transform: translateY(10px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }

        .animate-slide-down {
          animation: slide-down 0.3s ease-out;
        }

        .animate-slide-up {
          animation: slide-up 0.2s ease-out;
        }
      `}</style>
    </div>
  );
}
