"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import {
  Users,
  FileCheck,
  Clock,
  CheckCircle,
  XCircle,
  TrendingUp,
} from "lucide-react";

interface Statistics {
  total_users: number;
  pending_users: number;
  under_evaluation: number;
  evaluated_users: number;
  accepted_users: number;
  rejected_users: number;
  total_evaluations: number;
  avg_evaluation_score: number;
}

export default function AdminDashboard() {
  const [stats, setStats] = useState<Statistics | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState("");

  useEffect(() => {
    loadStatistics();
  }, []);

  const loadStatistics = async () => {
    try {
      const response = await adminAPI.getStatistics();
      setStats(response.data.data);
    } catch (err: any) {
      setError(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  if (error) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-4">
        <p className="text-red-800">{error}</p>
      </div>
    );
  }

  const statCards = [
    {
      title: "إجمالي المستخدمين",
      value: stats?.total_users || 0,
      icon: Users,
      color: "blue",
      gradient: "from-blue-500 to-blue-600",
    },
    {
      title: "قيد الانتظار",
      value: stats?.pending_users || 0,
      icon: Clock,
      color: "yellow",
      gradient: "from-yellow-500 to-yellow-600",
    },
    {
      title: "تحت التقييم",
      value: stats?.under_evaluation || 0,
      icon: FileCheck,
      color: "purple",
      gradient: "from-purple-500 to-purple-600",
    },
    {
      title: "تم التقييم",
      value: stats?.evaluated_users || 0,
      icon: CheckCircle,
      color: "indigo",
      gradient: "from-indigo-500 to-indigo-600",
    },
    {
      title: "المقبولين",
      value: stats?.accepted_users || 0,
      icon: CheckCircle,
      color: "green",
      gradient: "from-green-500 to-green-600",
    },
    {
      title: "المرفوضين",
      value: stats?.rejected_users || 0,
      icon: XCircle,
      color: "red",
      gradient: "from-red-500 to-red-600",
    },
  ];

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">لوحة التحكم</h1>
          <p className="text-gray-600 mt-1">نظرة عامة على النظام</p>
        </div>
      </div>

      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {statCards.map((card) => {
          const Icon = card.icon;
          return (
            <div
              key={card.title}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow"
            >
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">{card.title}</p>
                  <p className="text-3xl font-bold text-gray-900">
                    {card.value}
                  </p>
                </div>
                <div
                  className={`p-4 rounded-lg bg-gradient-to-br ${card.gradient}`}
                >
                  <Icon className="w-8 h-8 text-white" />
                </div>
              </div>
            </div>
          );
        })}
      </div>

      {/* Additional Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white rounded-xl shadow-lg p-6">
          <div className="flex items-center gap-3 mb-4">
            <div className="p-3 rounded-lg bg-gradient-to-br from-indigo-500 to-indigo-600">
              <TrendingUp className="w-6 h-6 text-white" />
            </div>
            <h3 className="text-lg font-semibold text-gray-900">
              إحصائيات التقييمات
            </h3>
          </div>
          <div className="space-y-3">
            <div className="flex justify-between items-center">
              <span className="text-gray-600">إجمالي التقييمات</span>
              <span className="text-xl font-bold text-gray-900">
                {stats?.total_evaluations || 0}
              </span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600">متوسط الدرجات</span>
              <span className="text-xl font-bold text-gray-900">
                {stats?.avg_evaluation_score?.toFixed(2) || "0.00"}
              </span>
            </div>
          </div>
        </div>

        <div className="bg-gradient-to-br from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-black">
          <h3 className="text-lg font-semibold mb-4">روابط سريعة</h3>
          <div className="space-y-2">
            <a
              href="/admin/users"
              className="block px-4 py-2 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors"
            >
              إدارة المستخدمين
            </a>
            <a
              href="/admin/categories"
              className="block px-4 py-2 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors"
            >
              إدارة التصنيفات
            </a>
            <a
              href="/admin/exams"
              className="block px-4 py-2 bg-white bg-opacity-20 rounded-lg hover:bg-opacity-30 transition-colors"
            >
              إدارة الاختبارات
            </a>
          </div>
        </div>
      </div>
    </div>
  );
}
