"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import {
  Activity,
  User,
  Calendar,
  Filter,
  Search,
  Eye,
  Download,
} from "lucide-react";
import { Button } from "@/components/ui/Button";

interface Log {
  id: number;
  user_id: number | null;
  user_name?: string;
  action: string;
  subject_type: string | null;
  subject_id: number | null;
  meta: Record<string, any> | null;
  ip_address?: string;
  user_agent?: string;
  created_at: string;
}

interface PaginationData {
  current_page: number;
  last_page: number;
  total: number;
  per_page: number;
  from: number;
  to: number;
}

export default function LogsPage() {
  const [logs, setLogs] = useState<Log[]>([]);
  const [loading, setLoading] = useState(true);
  const [statistics, setStatistics] = useState<any>(null);
  const [pagination, setPagination] = useState<PaginationData>({
    current_page: 1,
    last_page: 1,
    total: 0,
    per_page: 10,
    from: 0,
    to: 0,
  });
  const [filters, setFilters] = useState({
    action: "",
    user_id: "",
    date_from: "",
    date_to: "",
    search: "",
    page: 1,
  });
  const [actions, setActions] = useState<string[]>([]);
  const [selectedLog, setSelectedLog] = useState<Log | null>(null);

  useEffect(() => {
    loadLogs();
  }, [filters]);

  useEffect(() => {
    loadActions();
    loadStatistics();
  }, []);

  const loadLogs = async () => {
    setLoading(true);
    try {
      const response = await adminAPI.getLogs(filters);
      const data = response.data.data;

      // Extract logs from the data array
      const logsData = Array.isArray(data?.data) ? data.data : [];
      setLogs(logsData);

      // Extract pagination info
      setPagination({
        current_page: data?.current_page || 1,
        last_page: data?.last_page || 1,
        total: data?.total || 0,
        per_page: data?.per_page || 10,
        from: data?.from || 0,
        to: data?.to || 0,
      });
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const loadActions = async () => {
    try {
      const response = await adminAPI.getLogActions();
      setActions(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    }
  };

  const loadStatistics = async () => {
    try {
      const response = await adminAPI.getLogStatistics();
      setStatistics(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    }
  };

  const handleFilterChange = (key: string, value: string) => {
    setFilters({ ...filters, [key]: value });
  };

  const applyFilters = () => {
    loadLogs();
  };

  const getActionLabel = (action: string) => {
    const labels: Record<string, string> = {
      // Authentication
      "user.login": "تسجيل دخول",
      "user.login_attempt_while_locked": "تسجيل دخول والاكونت مغلق",
      "user.login_successful": "تسجيل دخول ناجح",
      "user.login_failed": "فشل تسجيل الدخول",
      "user.account_locked": "قفل الحساب",
      "user.logout": "تسجيل خروج",
      "user.password_changed": "تغيير كلمة المرور",

      // User Management
      "user.created": "إنشاء مستخدم",
      "user.accepted": "قبول مستخدم",
      "user.rejected": "رفض مستخدم",
      "user.categories_assigned": "تعيين تصنيفات",
      "user.categories_removed": "إزالة تصنيفات",

      // Registration
      "applicant.registered": "تسجيل متقدم",
      "registration.submitted": "إرسال تسجيل",

      // Exams
      "exam.created": "إنشاء اختبار",
      "exam.updated": "تحديث اختبار",
      "exam.deleted": "حذف اختبار",
      "exam.published": "نشر اختبار",
      "exam.unpublished": "إلغاء نشر اختبار",
      "exam.attempt_started": "بدء محاولة اختبار",
      "exam.attempt_completed": "إكمال محاولة اختبار",
      "exam.auto_submitted": "إرسال اختبار تلقائي",

      // Evaluations
      "evaluation_sheet.created": "إنشاء استمارة تقييم",
      "evaluation_sheet.deleted": "حذف استمارة تقييم",
      "evaluation.submitted": "إرسال تقييم",
      "evaluation.aggregated": "تجميع التقييمات",

      // Categories
      "category.created": "إنشاء تصنيف",
      "category.updated": "تحديث تصنيف",
      "category.deleted": "حذف تصنيف",
      "category.user_assigned": "تعيين مستخدم لتصنيف",
      "category.user_removed": "إزالة مستخدم من تصنيف",
      "category.bulk_users_assigned": "تعيين عدة مستخدمين لتصنيف",
      "category.bulk_users_removed": "إزالة عدة مستخدمين من تصنيف",

      // Settings
      "setting.updated": "تحديث الإعدادات",

      // Documents
      "document.downloaded": "تنزيل مستند",
    };
    return labels[action] || action;
  };

  const getActionCategory = (action: string) => {
    if (action.startsWith("user.")) return "المستخدمين";
    if (action.startsWith("exam.")) return "الاختبارات";
    if (action.startsWith("evaluation")) return "التقييمات";
    if (action.startsWith("category.")) return "التصنيفات";
    if (action.startsWith("applicant.") || action.startsWith("registration."))
      return "التسجيل";
    if (action.startsWith("setting.")) return "الإعدادات";
    if (action.startsWith("document.")) return "المستندات";
    return "أخرى";
  };

  const getMetadataLabel = (key: string): string => {
    const labels: Record<string, string> = {
      user_id: "معرف المستخدم",
      category_id: "معرف التصنيف",
      exam_id: "معرف الاختبار",
      email: "البريد الإلكتروني",
      name: "الاسم",
      title: "العنوان",
      score: "الدرجة",
      reason: "السبب",
      notes: "الملاحظات",
      count: "العدد",
      category_ids: "معرفات التصنيفات",
      user_ids: "معرفات المستخدمين",
      attempts: "المحاولات",
      lock_duration_minutes: "مدة القفل (دقائق)",
      ip: "عنوان IP",
      overall_score: "الدرجة الإجمالية",
      updated_settings: "الإعدادات المحدثة",
      evaluated_user_id: "معرف المستخدم المقيم",
      category_name: "اسم التصنيف",
    };
    return labels[key] || key.replace(/_/g, " ");
  };

  const formatMetaValue = (value: any): string | React.ReactNode => {
    if (typeof value === "boolean") return value ? "نعم" : "لا";
    if (Array.isArray(value)) return value.join(", ");
    if (typeof value === "object" && value !== null) {
      return JSON.stringify(value, null, 2);
    }
    return String(value);
  };

  const renderMetaValue = (value: any) => {
    if (typeof value === "boolean") return value ? "نعم" : "لا";
    if (Array.isArray(value)) return value.join(", ");
    if (typeof value === "object" && value !== null) {
      return (
        <pre className="bg-gray-800 text-gray-100 p-3 rounded text-xs overflow-auto max-h-64 border border-gray-700">
          {JSON.stringify(value, null, 2)}
        </pre>
      );
    }
    return String(value);
  };

  const goToPage = (page: number) => {
    if (page >= 1 && page <= pagination.last_page) {
      setFilters({ ...filters, page });
    }
  };

  const clearFilters = () => {
    setFilters({
      action: "",
      user_id: "",
      date_from: "",
      date_to: "",
      search: "",
      page: 1,
    });
    setTimeout(loadLogs, 100);
  };

  const getActionColor = (action: string) => {
    // Success actions (green)
    if (
      action.includes("accepted") ||
      action.includes("published") ||
      action.includes("completed") ||
      action.includes("successful") ||
      action.includes("submitted") ||
      action.includes("aggregated")
    ) {
      return "text-green-600 bg-green-50 border-green-200";
    }

    // Error/Negative actions (red)
    if (
      action.includes("rejected") ||
      action.includes("removed") ||
      action.includes("deleted") ||
      action.includes("failed") ||
      action.includes("locked")
    ) {
      return "text-red-600 bg-red-50 border-red-200";
    }

    // Creation/Start actions (blue)
    if (
      action.includes("created") ||
      action.includes("started") ||
      action.includes("registered") ||
      action.includes("assigned")
    ) {
      return "text-blue-600 bg-blue-50 border-blue-200";
    }

    // Update actions (purple)
    if (action.includes("updated") || action.includes("changed")) {
      return "text-purple-600 bg-purple-50 border-purple-200";
    }

    // Default
    return "text-gray-600 bg-gray-50 border-gray-200";
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  return (
    <div className="p-8">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900">سجل النظام</h1>
        <p className="text-gray-600 mt-2">
          تتبع جميع الأنشطة والعمليات في النظام
        </p>
      </div>

      {/* Statistics Cards */}
      {statistics && (
        <>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <div className="bg-white rounded-lg shadow p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm font-medium">
                    إجمالي السجلات
                  </p>
                  <p className="text-3xl font-bold text-gray-900 mt-2">
                    {statistics.total_logs}
                  </p>
                </div>
                <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                  <Activity className="w-6 h-6 text-blue-600" />
                </div>
              </div>
            </div>

            <div className="bg-white rounded-lg shadow p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm font-medium">اليوم</p>
                  <p className="text-3xl font-bold text-gray-900 mt-2">
                    {statistics.logs_today}
                  </p>
                </div>
                <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                  <Calendar className="w-6 h-6 text-green-600" />
                </div>
              </div>
            </div>

            <div className="bg-white rounded-lg shadow p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm font-medium">
                    هذا الأسبوع
                  </p>
                  <p className="text-3xl font-bold text-gray-900 mt-2">
                    {statistics.logs_this_week}
                  </p>
                </div>
                <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                  <Activity className="w-6 h-6 text-purple-600" />
                </div>
              </div>
            </div>

            <div className="bg-white rounded-lg shadow p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm font-medium">هذا الشهر</p>
                  <p className="text-3xl font-bold text-gray-900 mt-2">
                    {statistics.logs_this_month}
                  </p>
                </div>
                <div className="w-12 h-12 bg-orange-100 rounded-full flex items-center justify-center">
                  <Calendar className="w-6 h-6 text-orange-600" />
                </div>
              </div>
            </div>
          </div>

          {/* Most Common Actions */}
          {statistics.logs_by_action &&
            Object.keys(statistics.logs_by_action).length > 0 && (
              <div className="bg-white rounded-lg shadow p-6 mb-6">
                <h2 className="text-lg font-semibold text-gray-900 mb-4">
                  أكثر العمليات شيوعاً
                </h2>
                <div className="space-y-3">
                  {Object.entries(statistics.logs_by_action)
                    .sort(([, a]: any, [, b]: any) => b - a)
                    .slice(0, 5)
                    .map(([action, count]: [string, any]) => (
                      <div
                        key={action}
                        className="flex items-center justify-between pb-3 border-b border-gray-200 last:border-0"
                      >
                        <div className="flex items-center gap-2 flex-1">
                          <div className="flex-1">
                            <p className="text-sm font-medium text-gray-900">
                              {getActionLabel(action)}
                            </p>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <div className="w-32 bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-blue-600 h-2 rounded-full"
                              style={{
                                width: `${
                                  ((count as number) /
                                    Math.max(
                                      ...Object.values(
                                        statistics.logs_by_action as any
                                      ).map((v: any) => Number(v))
                                    )) *
                                  100
                                }%`,
                              }}
                            />
                          </div>
                          <span className="text-sm font-semibold text-gray-900 min-w-12">
                            {count}
                          </span>
                        </div>
                      </div>
                    ))}
                </div>
              </div>
            )}
        </>
      )}

      {/* Filters */}
      <div className="bg-white rounded-lg shadow p-6 mb-6">
        <div className="flex items-center gap-3 mb-4">
          <Filter className="w-5 h-5 text-gray-600" />
          <h2 className="text-lg font-semibold text-gray-900">تصفية السجلات</h2>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              نوع العملية
            </label>
            <select
              value={filters.action}
              onChange={(e) => handleFilterChange("action", e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
            >
              <option value="">جميع العمليات</option>
              {actions.map((action) => (
                <option key={action} value={action}>
                  {getActionLabel(action)}
                </option>
              ))}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              معرف المستخدم
            </label>
            <input
              type="number"
              value={filters.user_id}
              onChange={(e) => handleFilterChange("user_id", e.target.value)}
              placeholder="أدخل معرف المستخدم"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              من تاريخ
            </label>
            <input
              type="date"
              value={filters.date_from}
              onChange={(e) => handleFilterChange("date_from", e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              إلى تاريخ
            </label>
            <input
              type="date"
              value={filters.date_to}
              onChange={(e) => handleFilterChange("date_to", e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
            />
          </div>
        </div>

        <div className="mt-4">
          <label className="block text-sm font-medium text-gray-700 mb-2">
            بحث
          </label>
          <div className="flex gap-3">
            <div className="flex-1 relative">
              <Search className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                value={filters.search}
                onChange={(e) => handleFilterChange("search", e.target.value)}
                placeholder="ابحث في السجلات..."
                className="w-full pr-10 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
              />
            </div>
            <Button onClick={applyFilters}>تطبيق</Button>
            <Button variant="outline" onClick={clearFilters}>
              مسح
            </Button>
          </div>
        </div>
      </div>

      {/* Logs Table */}
      <div className="bg-white rounded-lg shadow">
        {logs.length === 0 ? (
          <div className="text-center py-12">
            <Activity className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <p className="text-gray-500">لا توجد سجلات</p>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b border-gray-200">
                <tr>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    المستخدم
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    العملية
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    التفاصيل
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    التاريخ
                  </th>
                  <th className="px-6 py-4 text-right text-sm font-semibold text-gray-900">
                    الإجراءات
                  </th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {logs.map((log) => (
                  <tr key={log.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4">
                      <div className="flex items-center gap-3">
                        <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                          <User className="w-4 h-4 text-blue-600" />
                        </div>
                        <div>
                          <p className="font-medium text-gray-900">
                            {log.user_name || "النظام"}
                          </p>
                          <p className="text-sm text-gray-500">
                            ID: {log.user_id || "-"}
                          </p>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <span
                        className={`inline-flex items-center gap-2 px-3 py-1 rounded-full text-sm font-medium border ${getActionColor(
                          log.action
                        )}`}
                      >
                        {getActionLabel(log.action)}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-600">
                        {log.subject_type && (
                          <p className="mb-2">
                            <span className="font-medium">النوع:</span>{" "}
                            <span className="text-gray-700">
                              {log.subject_type.split("\\").pop()}
                            </span>
                          </p>
                        )}
                        {log.subject_id && (
                          <p className="mb-2">
                            <span className="font-medium">المعرف:</span>{" "}
                            <span className="text-gray-700">
                              {log.subject_id}
                            </span>
                          </p>
                        )}
                        {log.meta && Object.keys(log.meta).length > 0 && (
                          <div className="mt-2 space-y-1">
                            {Object.entries(log.meta)
                              .slice(0, 2)
                              .map(([key, value]) => {
                                const isComplex =
                                  typeof value === "object" &&
                                  value !== null &&
                                  !Array.isArray(value) &&
                                  Object.keys(value).length > 2;
                                return (
                                  <div
                                    key={key}
                                    className="flex flex-col gap-1"
                                  >
                                    <span className="text-xs font-medium text-gray-500">
                                      {getMetadataLabel(key)}:
                                    </span>
                                    <div className="text-xs text-gray-700 break-words">
                                      {typeof value === "string" ? (
                                        value.length > 50 ? (
                                          <span>
                                            {value.substring(0, 50)}...
                                          </span>
                                        ) : (
                                          value
                                        )
                                      ) : isComplex ? (
                                        <span className="text-gray-500 italic">
                                          ({Object.keys(value).length} حقول)
                                        </span>
                                      ) : (
                                        String(formatMetaValue(value))
                                      )}
                                    </div>
                                  </div>
                                );
                              })}
                            {Object.keys(log.meta).length > 2 && (
                              <span className="text-xs text-gray-500 italic text-center">
                                +{Object.keys(log.meta).length - 2} حقول أخرى
                              </span>
                            )}
                          </div>
                        )}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center gap-2 text-sm text-gray-600">
                        <Calendar className="w-4 h-4" />
                        {new Date(log.created_at).toLocaleString("ar-EG")}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <button
                        onClick={() => setSelectedLog(log)}
                        className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
                        title="عرض التفاصيل"
                      >
                        <Eye className="w-5 h-5 text-blue-600" />
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {/* Pagination */}
        {logs.length > 0 && pagination.last_page > 1 && (
          <div className="flex items-center justify-between px-6 py-4 border-t border-gray-200">
            <div className="text-sm text-gray-600">
              عرض {pagination.from} إلى {pagination.to} من {pagination.total}{" "}
              سجل
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                onClick={() => goToPage(pagination.current_page - 1)}
                disabled={pagination.current_page === 1}
                className="px-3 py-2 text-sm"
              >
                السابق
              </Button>

              <div className="flex items-center gap-1">
                {Array.from(
                  { length: Math.min(5, pagination.last_page) },
                  (_, i) => {
                    let pageNum;
                    if (pagination.last_page <= 5) {
                      pageNum = i + 1;
                    } else if (pagination.current_page <= 3) {
                      pageNum = i + 1;
                    } else if (
                      pagination.current_page >=
                      pagination.last_page - 2
                    ) {
                      pageNum = pagination.last_page - 4 + i;
                    } else {
                      pageNum = pagination.current_page - 2 + i;
                    }
                    return (
                      <button
                        key={pageNum}
                        onClick={() => goToPage(pageNum)}
                        className={`px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                          pagination.current_page === pageNum
                            ? "bg-blue-600 text-white"
                            : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                        }`}
                      >
                        {pageNum}
                      </button>
                    );
                  }
                )}
              </div>

              <Button
                variant="outline"
                onClick={() => goToPage(pagination.current_page + 1)}
                disabled={pagination.current_page === pagination.last_page}
                className="px-3 py-2 text-sm"
              >
                التالي
              </Button>
            </div>
          </div>
        )}
      </div>

      {/* Detail Modal */}
      {selectedLog && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h2 className="text-2xl font-bold text-gray-900">
                  تفاصيل السجل
                </h2>
                <button
                  onClick={() => setSelectedLog(null)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <span className="text-2xl">×</span>
                </button>
              </div>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  المستخدم
                </label>
                <p className="text-gray-900">
                  {selectedLog.user_name || "النظام"} (ID:{" "}
                  {selectedLog.user_id || "-"})
                </p>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  العملية
                </label>
                <span
                  className={`inline-flex items-center gap-2 px-3 py-1 rounded-full text-sm font-medium border ${getActionColor(
                    selectedLog.action
                  )}`}
                >
                  {getActionLabel(selectedLog.action)}
                </span>
              </div>

              {selectedLog.subject_type && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    نوع الكيان
                  </label>
                  <p className="text-gray-900">{selectedLog.subject_type}</p>
                </div>
              )}

              {selectedLog.subject_id && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    معرف الكيان
                  </label>
                  <p className="text-gray-900">{selectedLog.subject_id}</p>
                </div>
              )}

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  التاريخ
                </label>
                <p className="text-gray-900">
                  {new Date(selectedLog.created_at).toLocaleString("ar-EG")}
                </p>
              </div>

              {selectedLog.ip_address && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    عنوان IP
                  </label>
                  <p className="text-gray-900 font-mono text-sm">
                    {selectedLog.ip_address}
                  </p>
                </div>
              )}

              {selectedLog.user_agent && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    وكيل المستخدم
                  </label>
                  <p className="text-gray-900 text-sm break-all">
                    {selectedLog.user_agent}
                  </p>
                </div>
              )}

              {selectedLog.meta && Object.keys(selectedLog.meta).length > 0 && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    البيانات الإضافية
                  </label>
                  <div className="space-y-3">
                    {Object.entries(selectedLog.meta).map(([key, value]) => (
                      <div
                        key={key}
                        className="border border-gray-200 rounded-lg p-4 bg-gray-50"
                      >
                        <div className="text-sm font-medium text-gray-700 mb-2">
                          {getMetadataLabel(key)}
                        </div>
                        <div className="text-sm text-gray-900">
                          {renderMetaValue(value)}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
            <div className="p-6 border-t border-gray-200">
              <Button
                onClick={() => setSelectedLog(null)}
                variant="outline"
                className="w-full"
              >
                إغلاق
              </Button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
