"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import { Plus, Edit2, Trash2, X, Search, Map } from "lucide-react";
import { Button } from "@/components/ui/Button";

interface Governorate {
  id: number;
  name_en: string;
  name_ar: string;
  code?: string;
  created_at: string;
}

export default function GovernoratesPage() {
  const [governorates, setGovernorate] = useState<Governorate[]>([]);
  const [filteredGovernorate, setFilteredGovernorate] = useState<Governorate[]>(
    []
  );
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [editingGovernorate, setEditingGovernorate] =
    useState<Governorate | null>(null);
  const [formData, setFormData] = useState({
    name_en: "",
    name_ar: "",
    code: "",
  });
  const [searchTerm, setSearchTerm] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [notification, setNotification] = useState<{
    show: boolean;
    message: string;
    type: "success" | "error";
  }>({ show: false, message: "", type: "success" });

  useEffect(() => {
    loadGovernorate();
  }, []);

  useEffect(() => {
    if (searchTerm.trim() === "") {
      setFilteredGovernorate(governorates);
    } else {
      const filtered = governorates.filter(
        (gov) =>
          gov.name_en.toLowerCase().includes(searchTerm.toLowerCase()) ||
          gov.name_ar.toLowerCase().includes(searchTerm.toLowerCase()) ||
          gov.code?.toLowerCase().includes(searchTerm.toLowerCase())
      );
      setFilteredGovernorate(filtered);
    }
  }, [searchTerm, governorates]);

  const showNotification = (message: string, type: "success" | "error") => {
    setNotification({ show: true, message, type });
    setTimeout(() => {
      setNotification({ show: false, message: "", type: "success" });
    }, 3000);
  };

  const loadGovernorate = async () => {
    try {
      const response = await adminAPI.getGovernorates();
      setGovernorate(response.data.data);
      setFilteredGovernorate(response.data.data);
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    } finally {
      setLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.name_en.trim() || !formData.name_ar.trim()) {
      showNotification("الاسم الإنجليزي والعربي مطلوبان", "error");
      return;
    }

    setIsSubmitting(true);
    try {
      if (editingGovernorate) {
        await adminAPI.updateGovernorate(editingGovernorate.id, formData);
        showNotification("تم تحديث المحافظة بنجاح", "success");
      } else {
        await adminAPI.createGovernorate(formData);
        showNotification("تم إنشاء المحافظة بنجاح", "success");
      }

      setShowModal(false);
      setFormData({ name_en: "", name_ar: "", code: "" });
      setEditingGovernorate(null);
      loadGovernorate();
    } catch (err: any) {
      showNotification(handleAPIError(err), "error");
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleEdit = (governorate: Governorate) => {
    setEditingGovernorate(governorate);
    setFormData({
      name_en: governorate.name_en,
      name_ar: governorate.name_ar,
      code: governorate.code || "",
    });
    setShowModal(true);
  };

  const handleDelete = async (id: number) => {
    if (confirm("هل أنت متأكد من حذف هذه المحافظة؟")) {
      try {
        await adminAPI.deleteGovernorate(id);
        showNotification("تم حذف المحافظة بنجاح", "success");
        loadGovernorate();
      } catch (err: any) {
        showNotification(handleAPIError(err), "error");
      }
    }
  };

  const closeModal = () => {
    setShowModal(false);
    setFormData({ name_en: "", name_ar: "", code: "" });
    setEditingGovernorate(null);
  };

  return (
    <div className="space-y-6" dir="rtl">
      {/* Notification */}
      {notification.show && (
        <div
          className={`p-4 rounded-lg ${
            notification.type === "success"
              ? "bg-green-50 text-green-800 border border-green-200"
              : "bg-red-50 text-red-800 border border-red-200"
          }`}
        >
          {notification.message}
        </div>
      )}

      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <div className="flex items-center gap-2">
            <Map className="w-8 h-8 text-blue-600" />
            <h1 className="text-3xl font-bold text-gray-900">المحافظات</h1>
          </div>
          <p className="text-gray-600 mt-1">إدارة المحافظات</p>
        </div>
        <Button
          onClick={() => setShowModal(true)}
          className="bg-blue-600 hover:bg-blue-700 text-white flex items-center gap-2"
        >
          <Plus className="w-4 h-4" />
          إضافة محافظة
        </Button>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute right-3 top-3 w-5 h-5 text-gray-400" />
        <input
          type="text"
          placeholder="ابحث عن محافظة..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          className="w-full pr-10 pl-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-right"
        />
      </div>

      {/* Governorates Table */}
      <div className="bg-white rounded-lg shadow overflow-hidden">
        {loading ? (
          <div className="flex items-center justify-center h-64">
            <div className="text-gray-500">جاري التحميل...</div>
          </div>
        ) : filteredGovernorate.length === 0 ? (
          <div className="flex items-center justify-center h-64">
            <div className="text-gray-500">لم يتم العثور على محافظات</div>
          </div>
        ) : (
          <table className="w-full">
            <thead className="bg-gray-50 border-b border-gray-200">
              <tr>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الاسم الإنجليزي
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الاسم العربي
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الرمز
                </th>
                <th className="px-6 py-3 text-right text-sm font-medium text-gray-700">
                  الإجراءات
                </th>
              </tr>
            </thead>
            <tbody>
              {filteredGovernorate.map((governorate) => (
                <tr
                  key={governorate.id}
                  className="border-b border-gray-200 hover:bg-gray-50"
                >
                  <td className="px-6 py-4 text-sm text-gray-900">
                    {governorate.name_en}
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-900 text-right">
                    {governorate.name_ar}
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-600">
                    {governorate.code || "-"}
                  </td>
                  <td className="px-6 py-4 text-sm">
                    <div className="flex items-center gap-2">
                      <button
                        onClick={() => handleEdit(governorate)}
                        className="p-2 text-blue-600 hover:bg-blue-50 rounded"
                      >
                        <Edit2 className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() => handleDelete(governorate.id)}
                        className="p-2 text-red-600 hover:bg-red-50 rounded"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </div>

      {/* Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div
            className="bg-white rounded-lg shadow-lg p-6 w-full max-w-md"
            dir="rtl"
          >
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl font-bold">
                {editingGovernorate ? "تعديل المحافظة" : "إضافة محافظة جديدة"}
              </h2>
              <button
                onClick={closeModal}
                className="text-gray-500 hover:text-gray-700"
              >
                <X className="w-5 h-5" />
              </button>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  الاسم الإنجليزي *
                </label>
                <input
                  type="text"
                  value={formData.name_en}
                  onChange={(e) =>
                    setFormData({ ...formData, name_en: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="مثل: القاهرة"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  الاسم العربي *
                </label>
                <input
                  type="text"
                  value={formData.name_ar}
                  onChange={(e) =>
                    setFormData({ ...formData, name_ar: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-right"
                  placeholder="مثل: القاهرة"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  الرمز (اختياري)
                </label>
                <input
                  type="text"
                  value={formData.code}
                  onChange={(e) =>
                    setFormData({ ...formData, code: e.target.value })
                  }
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="مثل: CA"
                />
              </div>

              <div className="flex gap-3 pt-4">
                <button
                  type="button"
                  onClick={closeModal}
                  className="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                >
                  إلغاء
                </button>
                <button
                  type="submit"
                  disabled={isSubmitting}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
                >
                  {isSubmitting ? "جاري الحفظ..." : "حفظ"}
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
}
