"use client";

import { useEffect, useState } from "react";
import { adminAPI, handleAPIError } from "@/lib/api";
import {
  Plus,
  Edit2,
  Trash2,
  Search,
  Eye,
  EyeOff,
  Copy,
  X,
  AlertCircle,
  BarChart3,
  FileText,
} from "lucide-react";

interface EvaluationSheet {
  id: number;
  title: string;
  criteria: Array<{ name: string; max_score: number }>;
  is_active: boolean;
  evaluations_count: number;
  created_by: number;
}

interface SheetTemplate {
  name: string;
  description: string;
  criteria: Array<{ name: string; max_score: number }>;
}

const TEMPLATES: SheetTemplate[] = [
  {
    name: "Leadership Assessment",
    description: "Evaluate leadership qualities and decision-making",
    criteria: [
      { name: "Communication Skills", max_score: 20 },
      { name: "Decision Making", max_score: 20 },
      { name: "Team Building", max_score: 20 },
      { name: "Strategic Thinking", max_score: 20 },
      { name: "Accountability", max_score: 20 },
    ],
  },
  {
    name: "Technical Competency",
    description: "Assess technical knowledge and problem-solving",
    criteria: [
      { name: "Technical Knowledge", max_score: 25 },
      { name: "Problem Solving", max_score: 25 },
      { name: "Attention to Detail", max_score: 25 },
      { name: "Innovation", max_score: 25 },
    ],
  },
  {
    name: "Diplomatic Skills",
    description: "Evaluate diplomatic and negotiation abilities",
    criteria: [
      { name: "Negotiation Skills", max_score: 20 },
      { name: "Cultural Awareness", max_score: 20 },
      { name: "Adaptability", max_score: 20 },
      { name: "Relationship Building", max_score: 20 },
      { name: "Conflict Resolution", max_score: 20 },
    ],
  },
  {
    name: "General Performance",
    description: "Basic performance evaluation",
    criteria: [
      { name: "Overall Performance", max_score: 30 },
      { name: "Potential", max_score: 30 },
      { name: "Professionalism", max_score: 40 },
    ],
  },
];

export default function EvaluationSheetsPage() {
  const [sheets, setSheets] = useState<EvaluationSheet[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [showModal, setShowModal] = useState(false);
  const [editingSheet, setEditingSheet] = useState<EvaluationSheet | null>(
    null
  );
  const [selectedTemplate, setSelectedTemplate] =
    useState<SheetTemplate | null>(null);
  const [showTemplates, setShowTemplates] = useState(false);
  const [viewingSheet, setViewingSheet] = useState<EvaluationSheet | null>(
    null
  );
  const [showViewModal, setShowViewModal] = useState(false);
  const [formData, setFormData] = useState({
    title: "",
    criteria: [{ name: "", max_score: 10 }],
    is_active: true,
  });
  const [submitting, setSubmitting] = useState(false);
  const [deleteConfirm, setDeleteConfirm] = useState<number | null>(null);

  useEffect(() => {
    loadSheets();
  }, []);

  const loadSheets = async () => {
    try {
      setLoading(true);
      const response = await adminAPI.getEvaluationSheets();
      setSheets(response.data.data);
    } catch (err: any) {
      console.error(handleAPIError(err));
    } finally {
      setLoading(false);
    }
  };

  const filteredSheets = sheets.filter((sheet) =>
    sheet.title.toLowerCase().includes(search.toLowerCase())
  );

  const openModal = (sheet?: EvaluationSheet) => {
    if (sheet) {
      setEditingSheet(sheet);
      setFormData({
        title: sheet.title,
        criteria: sheet.criteria,
        is_active: sheet.is_active,
      });
    } else {
      setEditingSheet(null);
      setFormData({
        title: "",
        criteria: [{ name: "", max_score: 10 }],
        is_active: true,
      });
    }
    setSelectedTemplate(null);
    setShowModal(true);
  };

  const closeModal = () => {
    setShowModal(false);
    setEditingSheet(null);
    setSelectedTemplate(null);
  };

  const applyTemplate = (template: SheetTemplate) => {
    setSelectedTemplate(template);
    setFormData({
      title: template.name,
      criteria: template.criteria,
      is_active: true,
    });
    setShowTemplates(false);
  };

  const handleTitleChange = (value: string) => {
    setFormData({ ...formData, title: value });
  };

  const handleCriteriaChange = (
    index: number,
    field: "name" | "max_score",
    value: any
  ) => {
    const newCriteria = [...formData.criteria];
    if (field === "max_score") {
      newCriteria[index][field] = Number(value);
    } else {
      newCriteria[index][field] = value;
    }
    setFormData({ ...formData, criteria: newCriteria });
  };

  const addCriterion = () => {
    setFormData({
      ...formData,
      criteria: [...formData.criteria, { name: "", max_score: 10 }],
    });
  };

  const removeCriterion = (index: number) => {
    setFormData({
      ...formData,
      criteria: formData.criteria.filter((_, i) => i !== index),
    });
  };

  const handleSubmit = async () => {
    if (!formData.title.trim()) {
      alert("يرجى إدخال عنوان النموذج");
      return;
    }

    if (formData.criteria.some((c) => !c.name.trim())) {
      alert("يرجى ملء جميع أسماء المعايير");
      return;
    }

    setSubmitting(true);
    try {
      if (editingSheet) {
        await adminAPI.updateEvaluationSheet(editingSheet.id, formData);
      } else {
        await adminAPI.createEvaluationSheet(formData);
      }
      await loadSheets();
      closeModal();
    } catch (err: any) {
      alert(handleAPIError(err));
    } finally {
      setSubmitting(false);
    }
  };

  const handleDelete = async (id: number) => {
    setSubmitting(true);
    try {
      await adminAPI.deleteEvaluationSheet(id);
      await loadSheets();
      setDeleteConfirm(null);
    } catch (err: any) {
      alert(handleAPIError(err));
    } finally {
      setSubmitting(false);
    }
  };

  const toggleActive = async (sheet: EvaluationSheet) => {
    setSubmitting(true);
    try {
      await adminAPI.updateEvaluationSheet(sheet.id, {
        is_active: !sheet.is_active,
      });
      await loadSheets();
    } catch (err: any) {
      alert(handleAPIError(err));
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin" />
      </div>
    );
  }

  const totalMaxScore = formData.criteria.reduce(
    (sum, c) => sum + c.max_score,
    0
  );

  return (
    <div className="space-y-6" dir="rtl">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">نماذج التقييم</h1>
          <p className="text-gray-600 mt-1">
            إدارة نماذج التقييم المستخدمة من قبل المقيمين
          </p>
        </div>
        <button
          onClick={() => {
            openModal();
            setShowTemplates(true);
          }}
          className="flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg hover:from-blue-700 hover:to-indigo-700 transition-colors font-medium"
        >
          <Plus className="w-5 h-5" />
          نموذج جديد
        </button>
      </div>

      {/* Search */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="relative">
          <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
          <input
            type="text"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            placeholder="البحث عن النموذج..."
            className="w-full pr-11 pl-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>
      </div>

      {/* Sheets List */}
      {filteredSheets.length === 0 ? (
        <div className="bg-white rounded-xl shadow-lg p-12 text-center">
          <AlertCircle className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-500">
            {search ? "لا توجد نماذج تطابق البحث" : "لا توجد نماذج تقييم"}
          </p>
        </div>
      ) : (
        <div className="space-y-4">
          {filteredSheets.map((sheet) => (
            <div
              key={sheet.id}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow"
            >
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-xl font-semibold text-gray-900">
                      {sheet.title}
                    </h3>
                    {sheet.is_active && (
                      <span className="inline-block px-3 py-1 bg-green-100 text-green-700 text-sm font-medium rounded-full">
                        نشط
                      </span>
                    )}
                    {!sheet.is_active && (
                      <span className="inline-block px-3 py-1 bg-gray-100 text-gray-700 text-sm font-medium rounded-full">
                        غير نشط
                      </span>
                    )}
                  </div>
                  <p className="text-sm text-gray-600">
                    {sheet.criteria.length} معايير • الدرجة الإجمالية:{" "}
                    {sheet.criteria.reduce((sum, c) => sum + c.max_score, 0)}
                  </p>
                </div>
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => {
                      setViewingSheet(sheet);
                      setShowViewModal(true);
                    }}
                    className="p-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors"
                    title="عرض التفاصيل"
                  >
                    <FileText className="w-5 h-5" />
                  </button>
                  <button
                    onClick={() => toggleActive(sheet)}
                    disabled={submitting}
                    className="p-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors disabled:opacity-50"
                    title={sheet.is_active ? "تعطيل" : "تفعيل"}
                  >
                    {sheet.is_active ? (
                      <Eye className="w-5 h-5" />
                    ) : (
                      <EyeOff className="w-5 h-5" />
                    )}
                  </button>
                  <button
                    onClick={() => openModal(sheet)}
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                    title="تعديل"
                  >
                    <Edit2 className="w-5 h-5" />
                  </button>
                  <button
                    onClick={() => setDeleteConfirm(sheet.id)}
                    disabled={sheet.evaluations_count > 0}
                    className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                    title={
                      sheet.evaluations_count > 0
                        ? "لا يمكن حذف النموذج المستخدم"
                        : "حذف"
                    }
                  >
                    <Trash2 className="w-5 h-5" />
                  </button>
                </div>
              </div>

              {/* Criteria Preview */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                {sheet.criteria.map((criterion, index) => (
                  <div
                    key={index}
                    className="flex items-center justify-between p-3 bg-gray-200 rounded-lg"
                  >
                    <span className="text-sm font-medium text-gray-900">
                      {criterion.name}
                    </span>
                    <span className="text-sm text-gray-600">
                      الحد الأقصى: {criterion.max_score}
                    </span>
                  </div>
                ))}
              </div>

              {sheet.evaluations_count > 0 && (
                <div className="mt-4 p-3 bg-blue-50 rounded-lg flex items-center gap-2 text-sm text-blue-700">
                  <BarChart3 className="w-4 h-4" />
                  تم استخدام هذا النموذج في {sheet.evaluations_count} تقييم
                </div>
              )}
            </div>
          ))}
        </div>
      )}

      {/* Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="flex items-center justify-between p-6 border-b border-gray-200">
              <h2 className="text-2xl font-bold text-gray-900">
                {editingSheet ? "تعديل النموذج" : "نموذج جديد"}
              </h2>
              <button
                onClick={closeModal}
                disabled={submitting}
                className="p-2 text-gray-400 hover:text-gray-600 disabled:opacity-50"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="p-6 space-y-6">
              {/* Summary Banner */}
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-600 font-medium">
                    معلومات النموذج
                  </p>
                  <p className="text-2xl font-bold text-blue-900">
                    {formData.criteria.length} معايير • {totalMaxScore} درجة
                  </p>
                </div>
                {editingSheet && (
                  <div className="text-right">
                    <p className="text-xs text-gray-600">معرف النموذج</p>
                    <p className="text-lg font-semibold text-gray-900">
                      #{editingSheet.id}
                    </p>
                  </div>
                )}
              </div>

              {/* Template Selection */}
              {!editingSheet && showTemplates && (
                <div className="space-y-3">
                  <div className="flex items-center justify-between mb-4">
                    <h3 className="text-lg font-semibold text-gray-900">
                      اختر قالب
                    </h3>
                    <button
                      onClick={() => setShowTemplates(false)}
                      className="text-sm text-blue-600 hover:text-blue-700"
                    >
                      أو أنشئ من الصفر
                    </button>
                  </div>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                    {TEMPLATES.map((template, idx) => (
                      <button
                        key={idx}
                        onClick={() => applyTemplate(template)}
                        className={`p-4 border-2 rounded-lg text-right transition-all ${
                          selectedTemplate?.name === template.name
                            ? "border-blue-600 bg-blue-50"
                            : "border-gray-200 hover:border-blue-300"
                        }`}
                      >
                        <p className="font-semibold text-gray-900 mb-1">
                          {template.name}
                        </p>
                        <p className="text-sm text-gray-600">
                          {template.description}
                        </p>
                        <p className="text-xs text-gray-500 mt-2">
                          {template.criteria.length} معايير
                        </p>
                      </button>
                    ))}
                  </div>
                  {selectedTemplate && (
                    <div className="bg-blue-50 p-4 rounded-lg text-sm text-blue-700">
                      تم تحميل القالب: {selectedTemplate.name}
                    </div>
                  )}
                </div>
              )}

              {/* Form */}
              <div className="space-y-4">
                {/* Title */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    عنوان النموذج
                  </label>
                  <input
                    type="text"
                    value={formData.title}
                    onChange={(e) => handleTitleChange(e.target.value)}
                    placeholder="مثال: تقييم المهارات القيادية"
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>

                {/* Criteria */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    المعايير ({formData.criteria.length})
                  </label>
                  <div className="space-y-3 max-h-96 overflow-y-auto pr-2">
                    {formData.criteria.map((criterion, idx) => (
                      <div
                        key={idx}
                        className="flex gap-3 items-start bg-gray-50 p-3 rounded-lg"
                      >
                        <div className="flex-1">
                          <input
                            type="text"
                            value={criterion.name}
                            onChange={(e) =>
                              handleCriteriaChange(idx, "name", e.target.value)
                            }
                            placeholder="اسم المعيار"
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                          />
                        </div>
                        <div className="w-20">
                          <input
                            type="number"
                            value={criterion.max_score}
                            onChange={(e) =>
                              handleCriteriaChange(
                                idx,
                                "max_score",
                                e.target.value
                              )
                            }
                            placeholder="الدرجة"
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                          />
                        </div>
                        {formData.criteria.length > 1 && (
                          <button
                            onClick={() => removeCriterion(idx)}
                            className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                            type="button"
                          >
                            <X className="w-4 h-4" />
                          </button>
                        )}
                      </div>
                    ))}
                  </div>

                  <button
                    onClick={addCriterion}
                    className="mt-3 flex items-center gap-2 px-4 py-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors text-sm font-medium"
                    type="button"
                  >
                    <Plus className="w-4 h-4" />
                    إضافة معيار
                  </button>
                </div>

                {/* Active Status */}
                <div className="flex items-center gap-3">
                  <input
                    type="checkbox"
                    id="is_active"
                    checked={formData.is_active}
                    onChange={(e) =>
                      setFormData({
                        ...formData,
                        is_active: e.target.checked,
                      })
                    }
                    className="w-5 h-5 text-blue-600 rounded focus:ring-2 focus:ring-blue-500"
                  />
                  <label
                    htmlFor="is_active"
                    className="text-sm font-medium text-gray-700"
                  >
                    تفعيل النموذج (متاح للمقيمين)
                  </label>
                </div>
              </div>
            </div>

            {/* Actions */}
            <div className="flex gap-3 p-6 border-t border-gray-200">
              <button
                onClick={closeModal}
                disabled={submitting}
                className="flex-1 px-4 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium disabled:opacity-50"
              >
                إلغاء
              </button>
              <button
                onClick={handleSubmit}
                disabled={submitting}
                className="flex-1 px-4 py-3 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg hover:from-blue-700 hover:to-indigo-700 transition-colors font-medium disabled:opacity-50"
              >
                {submitting ? "جاري الحفظ..." : "حفظ"}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Delete Confirmation */}
      {deleteConfirm && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl max-w-sm w-full p-6 space-y-4">
            <div className="flex items-center gap-3 text-red-600">
              <AlertCircle className="w-6 h-6" />
              <h3 className="text-lg font-semibold">تأكيد الحذف</h3>
            </div>
            <p className="text-gray-600">
              هل أنت متأكد من حذف هذا النموذج؟ لا يمكن التراجع عن هذه العملية.
            </p>
            <div className="flex gap-3">
              <button
                onClick={() => setDeleteConfirm(null)}
                disabled={submitting}
                className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium disabled:opacity-50"
              >
                إلغاء
              </button>
              <button
                onClick={() => handleDelete(deleteConfirm)}
                disabled={submitting}
                className="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors font-medium disabled:opacity-50"
              >
                {submitting ? "جاري..." : "حذف"}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* View Details Modal */}
      {showViewModal && viewingSheet && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="flex items-center justify-between p-6 border-b border-gray-200">
              <h2 className="text-2xl font-bold text-gray-900">
                تفاصيل النموذج
              </h2>
              <button
                onClick={() => setShowViewModal(false)}
                className="p-2 text-gray-400 hover:text-gray-600"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="p-6 space-y-6">
              {/* Header Info */}
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <h3 className="text-2xl font-bold text-gray-900 mb-4">
                  {viewingSheet.title}
                </h3>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  <div>
                    <p className="text-xs text-gray-600 font-medium">
                      معرف النموذج
                    </p>
                    <p className="text-lg font-bold text-gray-900">
                      #{viewingSheet.id}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs text-gray-600 font-medium">الحالة</p>
                    <p className="mt-1">
                      {viewingSheet.is_active ? (
                        <span className="inline-block px-3 py-1 bg-green-100 text-green-700 text-sm font-medium rounded-full">
                          نشط
                        </span>
                      ) : (
                        <span className="inline-block px-3 py-1 bg-gray-100 text-gray-700 text-sm font-medium rounded-full">
                          غير نشط
                        </span>
                      )}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs text-gray-600 font-medium">
                      عدد المعايير
                    </p>
                    <p className="text-lg font-bold text-gray-900">
                      {viewingSheet.criteria.length}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs text-gray-600 font-medium">
                      الدرجة الإجمالية
                    </p>
                    <p className="text-lg font-bold text-blue-600">
                      {viewingSheet.criteria.reduce(
                        (sum, c) => sum + c.max_score,
                        0
                      )}
                    </p>
                  </div>
                </div>
              </div>

              {/* Criteria List */}
              <div>
                <h4 className="text-lg font-semibold text-gray-900 mb-4">
                  المعايير ({viewingSheet.criteria.length})
                </h4>
                <div className="space-y-3">
                  {viewingSheet.criteria.map((criterion, idx) => (
                    <div
                      key={idx}
                      className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border border-gray-200 hover:border-blue-300 transition-colors"
                    >
                      <div className="flex-1">
                        <p className="text-sm font-medium text-gray-900">
                          {idx + 1}. {criterion.name}
                        </p>
                      </div>
                      <div className="flex items-center gap-4">
                        <div className="text-right">
                          <p className="text-xs text-gray-600">الحد الأقصى</p>
                          <p className="text-2xl font-bold text-blue-600">
                            {criterion.max_score}
                          </p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              {/* Usage Stats */}
              {viewingSheet.evaluations_count > 0 && (
                <div className="bg-green-50 border border-green-200 rounded-lg p-4 flex items-center gap-3">
                  <BarChart3 className="w-5 h-5 text-green-600 flex-shrink-0" />
                  <div>
                    <p className="text-sm font-medium text-green-900">
                      تم استخدام هذا النموذج في
                    </p>
                    <p className="text-2xl font-bold text-green-600">
                      {viewingSheet.evaluations_count} تقييم
                    </p>
                  </div>
                </div>
              )}
            </div>

            {/* Actions */}
            <div className="flex gap-3 p-6 border-t border-gray-200">
              <button
                onClick={() => setShowViewModal(false)}
                className="flex-1 px-4 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
              >
                إغلاق
              </button>
              <button
                onClick={() => {
                  setShowViewModal(false);
                  openModal(viewingSheet);
                }}
                className="flex-1 px-4 py-3 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg hover:from-blue-700 hover:to-indigo-700 transition-colors font-medium"
              >
                تعديل النموذج
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
