<?php

namespace App\Services;

use Carbon\Carbon;

class SpecYearService
{
    /**
     * Determine form and spec_year based on National ID
     */
    public function determineFormByNationalId(string $nationalId): array
    {
        // Extract birthdate from 14-digit national ID
        // Format: YYMMDDHHHHHHSC (Simplified - adjust based on actual format)
        $birthdate = $this->extractBirthdateFromNationalId($nationalId);
        $age = $birthdate->diffInYears(Carbon::now());
        
        $rules = config('spec_year.rules', []);
        
        foreach ($rules as $rule) {
            $minAge = $rule['min_age'];
            $maxAge = $rule['max_age'];
            
            if ($age >= $minAge && ($maxAge === null || $age <= $maxAge)) {
                return [
                    'form_id' => $rule['form_id'],
                    'spec_year' => $rule['spec_year'],
                    'birthdate' => $birthdate->format('Y-m-d'),
                    'age' => $age,
                    'prefill' => $this->getPrefillData($nationalId, $birthdate),
                ];
            }
        }
        
        throw new \Exception('No matching form found for this age group');
    }

    private function extractBirthdateFromNationalId(string $nationalId): Carbon
    {
        // Assuming 14-digit format: YYMMDDHHHHHHSC
        // First 6 digits are YYMMDD
        $year = substr($nationalId, 0, 2);
        $month = substr($nationalId, 2, 2);
        $day = substr($nationalId, 4, 2);
        
        // Handle century (19xx vs 20xx)
        $fullYear = (int)$year < 50 ? 2000 + (int)$year : 1900 + (int)$year;
        
        return Carbon::createFromDate($fullYear, $month, $day);
    }

    private function getPrefillData(string $nationalId, Carbon $birthdate): array
    {
        // Return prefill data that can be extracted from national ID
        return [
            'birthdate' => $birthdate->format('Y-m-d'),
            // Other fields would be extracted from additional services/APIs
            // For now, return basic structure
        ];
    }
}
