<?php

namespace App\Services;

use App\Models\Registration;
use App\Models\User;
use App\Events\UserRegistered;
use Illuminate\Support\Facades\Hash;

class RegistrationService
{
    public function __construct(
        protected SpecYearService $specYearService,
        protected AuditService $auditService,
    ) {}

    public function createRegistration(array $data): Registration
    {
        // Extract spec_year from form_id (e.g., 'form_2024' -> '2024')
        $specYear = $this->extractSpecYearFromFormId($data['form_id']);

        $registration = Registration::create([
            'form_id' => $data['form_id'],
            'national_id' => encrypt($data['national_id']),
            'spec_year' => $specYear,
            'form_data' => $data['fields'],
            'status' => 'submitted',
            'submitted_at' => now(),
        ]);

        $user = $this->createOrUpdateUser($data, $registration);
        $registration->user_id = $user->id;
        $registration->save();

        event(new UserRegistered($user, $registration));

        $this->auditService->log(
            causer: $user,
            action: 'registration.submitted',
            subject: $registration,
        );

        return $registration;
    }

    private function createOrUpdateUser(array $data, Registration $registration): User
    {
        $email = $data['fields']['email'] ?? null;
        $password = $data['fields']['password'] ?? str()->random(12);

        $user = User::firstOrCreate(
            ['email' => $email],
            [
                'national_id' => encrypt($data['national_id']),
                'full_name' => $data['fields']['full_name'] ?? null,
                'role' => 'user',
                'is_active' => true,
                'password' => Hash::make($password),
            ]
        );

        return $user;
    }

    public function autosaveDraft(User $user, string $formId, array $fields): void
    {
        // Store draft in cache or database
        cache()->put(
            "registration_draft_{$user->id}_{$formId}",
            $fields,
            now()->addDays(30)
        );
    }

    public function getDraft(User $user, string $formId): ?array
    {
        return cache()->get("registration_draft_{$user->id}_{$formId}");
    }

    private function extractSpecYearFromFormId(string $formId): string
    {
        // Extract year from form_id like 'form_a_2024' or 'form_2024'
        // If no year found, use current year
        if (preg_match('/(\d{4})/', $formId, $matches)) {
            return $matches[1];
        }
        return (string) now()->year;
    }
}
