<?php

namespace App\Services;

use App\Models\Category;
use App\Models\User;
use Illuminate\Support\Collection;

class CategoryService
{
  public function __construct(
    protected AuditService $auditService,
  ) {}

  /**
   * Create a new category
   */
  public function createCategory(array $data, User $admin): Category
  {
    $category = Category::create([
      'name' => $data['name'],
      'description' => $data['description'] ?? null,
    ]);

    $this->auditService->log(
      causer: $admin,
      action: 'category.created',
      subject: $category,
      meta: ['name' => $category->name]
    );

    return $category;
  }

  /**
   * Assign a user to a category
   */
  public function assignUserToCategory(User $user, Category $category, User $admin): void
  {
    if (!$user->categories->contains($category->id)) {
      $user->categories()->attach($category->id);

      $this->auditService->log(
        causer: $admin,
        action: 'category.user_assigned',
        subject: $user,
        meta: [
          'category_id' => $category->id,
          'category_name' => $category->name,
          'user_id' => $user->id,
        ]
      );
    }
  }

  /**
   * Remove a user from a category
   */
  public function removeUserFromCategory(User $user, Category $category, User $admin): void
  {
    $user->categories()->detach($category->id);

    $this->auditService->log(
      causer: $admin,
      action: 'category.user_removed',
      subject: $user,
      meta: [
        'category_id' => $category->id,
        'category_name' => $category->name,
        'user_id' => $user->id,
      ]
    );
  }

  /**
   * Bulk assign users to a category
   */
  public function bulkAssignUsersToCategory(array $userIds, Category $category, User $admin): int
  {
    $count = 0;

    foreach ($userIds as $userId) {
      $user = User::find($userId);
      if ($user && !$user->categories->contains($category->id)) {
        $user->categories()->attach($category->id);
        $count++;
      }
    }

    $this->auditService->log(
      causer: $admin,
      action: 'category.bulk_users_assigned',
      subject: $category,
      meta: [
        'category_id' => $category->id,
        'user_count' => $count,
        'user_ids' => $userIds,
      ]
    );

    return $count;
  }

  /**
   * Bulk remove users from a category
   */
  public function bulkRemoveUsersFromCategory(array $userIds, Category $category, User $admin): int
  {
    $count = 0;

    foreach ($userIds as $userId) {
      $user = User::find($userId);
      if ($user && $user->categories->contains($category->id)) {
        $user->categories()->detach($category->id);
        $count++;
      }
    }

    $this->auditService->log(
      causer: $admin,
      action: 'category.bulk_users_removed',
      subject: $category,
      meta: [
        'category_id' => $category->id,
        'user_count' => $count,
        'user_ids' => $userIds,
      ]
    );

    return $count;
  }

  /**
   * Get all users in a category
   */
  public function getCategoryUsers(Category $category): Collection
  {
    return $category->users()->with(['applicant', 'examAttempts', 'evaluationAggregate'])->get();
  }

  /**
   * Get all categories for a user
   */
  public function getUserCategories(User $user): Collection
  {
    return $user->categories;
  }
}
