<?php

namespace App\Models;

use Illuminate\Auth\Authenticatable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Model
{
    use HasFactory, Notifiable, HasApiTokens, SoftDeletes;
    protected $fillable = [
        'email',
        'password',
        'national_id',
        'full_name',
        'role',
        'is_active',
        'last_login_at',
        'login_attempts',
        'locked_until',
        'status',
    ];

    protected $hidden = [
        'password',
        'national_id',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'last_login_at' => 'datetime',
        'locked_until' => 'datetime',
        'email_verified_at' => 'datetime',
    ];

    public function applicant(): HasOne
    {
        return $this->hasOne(Applicant::class);
    }

    public function examAttempts(): HasMany
    {
        return $this->hasMany(ExamAttempt::class);
    }

    public function evaluations(): HasMany
    {
        return $this->hasMany(Evaluation::class, 'evaluated_user_id');
    }

    public function systemLogs(): HasMany
    {
        return $this->hasMany(SystemLog::class, 'causer_id');
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(Category::class);
    }

    public function evaluationAggregate(): HasOne
    {
        return $this->hasOne(EvaluationAggregate::class, 'evaluated_user_id');
    }

    public function createdEvaluationSheets(): HasMany
    {
        return $this->hasMany(EvaluationSheet::class, 'created_by');
    }

    public function givenEvaluations(): HasMany
    {
        return $this->hasMany(Evaluation::class, 'judge_id');
    }

    // Helper methods
    public function hasCompletedExam(): bool
    {
        return $this->examAttempts()->whereNotNull('completed_at')->exists();
    }

    public function getEvaluationCount(): int
    {
        return $this->evaluations()->whereNotNull('submitted_at')->count();
    }

    public function hasMaxEvaluations(): bool
    {
        return $this->getEvaluationCount() >= 5;
    }

    public function isEvaluated(): bool
    {
        return $this->evaluationAggregate()->exists();
    }
}
