<?php

namespace App\Http\Controllers;

use App\Models\SystemLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class SystemLogController extends Controller
{
  /**
   * Get system logs with filters
   */
  public function index(Request $request): JsonResponse
  {
    $query = SystemLog::query()
      ->with('causer:id,full_name,email')
      ->orderBy('created_at', 'desc');

    // Filter by action
    if ($request->has('action') && $request->action) {
      $query->where('action', $request->action);
    }

    // Filter by causer (user_id in frontend)
    if ($request->has('user_id') && $request->user_id) {
      $query->where('causer_id', $request->user_id);
    }

    // Filter by date range
    if ($request->filled('date_from')) {
      $query->whereDate('created_at', '>=', $request->date_from);
    }

    if ($request->filled('date_to')) {
      $query->whereDate('created_at', '<=', $request->date_to);
    }

    // Search in action or meta
    if ($request->filled('search')) {
      $search = $request->search;
      $query->where(function ($q) use ($search) {
        $q->where('action', 'like', "%{$search}%")
          ->orWhere('meta', 'like', "%{$search}%");
      });
    }

    $logs = $query->paginate($request->per_page ?? 10);

    // Transform logs to include user_name
    $logs->getCollection()->transform(function ($log) {
      return [
        'id' => $log->id,
        'user_id' => $log->causer_id,
        'user_name' => $log->causer ? ($log->causer->full_name ?? $log->causer->email) : 'System',
        'action' => $log->action,
        'subject_type' => $log->subject_type,
        'subject_id' => $log->subject_id,
        'meta' => $log->meta,
        'ip_address' => $log->ip_address ?? null,
        'user_agent' => $log->user_agent ?? null,
        'created_at' => $log->created_at,
      ];
    });

    return response()->json([
      'success' => true,
      'data' => $logs,
    ]);
  }

  /**
   * Get logs for a specific user
   */
  public function userLogs(int $userId): JsonResponse
  {
    $logs = SystemLog::where(function ($query) use ($userId) {
      $query->where('causer_id', $userId)
        ->orWhere(function ($q) use ($userId) {
          $q->where('subject_type', 'App\\Models\\User')
            ->where('subject_id', $userId);
        });
    })
      ->orderBy('created_at', 'desc')
      ->paginate(50);

    return response()->json([
      'success' => true,
      'data' => $logs,
    ]);
  }

  /**
   * Get available log actions
   */
  public function actions(): JsonResponse
  {
    $actions = SystemLog::distinct('action')->pluck('action');

    return response()->json([
      'success' => true,
      'data' => $actions,
    ]);
  }

  /**
   * Get log statistics
   */
  public function statistics(Request $request): JsonResponse
  {
    $query = SystemLog::query();

    if ($request->has('from_date')) {
      $query->where('created_at', '>=', $request->from_date);
    }

    if ($request->has('to_date')) {
      $query->where('created_at', '<=', $request->to_date);
    }

    $stats = [
      'total_logs' => $query->count(),
      'logs_by_action' => (clone $query)->select('action', DB::raw('count(*) as count'))
        ->groupBy('action')
        ->get()
        ->pluck('count', 'action'),
      'logs_today' => (clone $query)->whereDate('created_at', today())->count(),
      'logs_this_week' => (clone $query)->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
      'logs_this_month' => (clone $query)->whereMonth('created_at', now()->month)->count(),
    ];

    return response()->json([
      'success' => true,
      'data' => $stats,
    ]);
  }
}
