<?php

namespace App\Http\Controllers;

use App\Models\Exam;
use App\Models\ExamAttempt;
use App\Services\ExamService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class ExamController extends Controller
{
  public function __construct(
    protected ExamService $examService,
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Get all published exams
   */
  public function index(): JsonResponse
  {
    $user = Auth::user();
    $durationMinutes = \App\Models\Setting::get('exam_duration', 60);
    $autoPublishResults = \App\Models\Setting::get('auto_publish_results', false);
    $passingScore = \App\Models\Setting::get('passing_score', 70);

    $exams = Exam::where('is_published', true)
      ->select(['id', 'title', 'description', 'published_at', 'deadline_at'])
      ->get()
      ->map(function ($exam) use ($user, $durationMinutes, $autoPublishResults, $passingScore) {
        $attempt = ExamAttempt::where('exam_id', $exam->id)
          ->where('user_id', $user->id)
          ->first();

        $hasCompleted = $attempt && $attempt->completed_at !== null;

        $examData = [
          'id' => $exam->id,
          'title' => $exam->title,
          'description' => $exam->description,
          'published_at' => $exam->published_at,
          'deadline_at' => $exam->deadline_at,
          'duration_minutes' => $durationMinutes,
          'is_completed' => $hasCompleted,
        ];

        // Include results if completed and auto-publish is enabled
        if ($hasCompleted && $autoPublishResults && $attempt) {
          $examData['results'] = [
            'score' => $attempt->score,
            'passing_score' => $passingScore,
            'passed' => $attempt->score >= $passingScore,
            'completed_at' => $attempt->completed_at,
            'auto_submitted' => $attempt->auto_submitted,
          ];
        }

        return $examData;
      });

    return response()->json([
      'success' => true,
      'data' => $exams,
    ]);
  }

  /**
   * Get a specific exam
   */
  public function show(Exam $exam): JsonResponse
  {
    $user = Auth::user();
    $attempt = $this->examService->getUserExamAttempt($exam, $user);
    $canTake = $this->examService->canUserTakeExam($exam, $user);
    $durationMinutes = \App\Models\Setting::get('exam_duration', 60);

    $examData = $exam->toArray();
    $examData['duration_minutes'] = $durationMinutes;

    // Remove correct answers from schema
    if (isset($examData['form_schema']['questions'])) {
      $examData['form_schema']['questions'] = array_map(function ($question) {
        unset($question['correct_answer']);
        return $question;
      }, $examData['form_schema']['questions']);
    }

    $responseData = [
      'exam' => $examData,
      'attempt' => $attempt,
      'can_take' => $canTake,
    ];

    // If user has completed exam and results are published, include results
    if ($attempt && $attempt->completed_at) {
      $autoPublishResults = \App\Models\Setting::get('auto_publish_results', false);
      if ($autoPublishResults) {
        $passingScore = \App\Models\Setting::get('passing_score', 70);
        $responseData['results'] = [
          'score' => $attempt->score,
          'passing_score' => $passingScore,
          'passed' => $attempt->score >= $passingScore,
          'completed_at' => $attempt->completed_at,
          'auto_submitted' => $attempt->auto_submitted,
        ];
      }
    }

    return response()->json([
      'success' => true,
      'data' => $responseData,
    ]);
  }

  /**
   * Start an exam attempt
   */
  public function start(Exam $exam): JsonResponse
  {
    try {
      $user = Auth::user();
      $attempt = $this->examService->startExamAttempt($exam, $user);

      return response()->json([
        'success' => true,
        'message' => 'Exam attempt started successfully',
        'data' => $attempt,
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Submit exam answers
   */
  public function submit(Request $request, Exam $exam): JsonResponse
  {
    $request->validate([
      'answers' => 'array',
      'auto_submit' => 'boolean',
    ]);

    try {
      $user = Auth::user();
      $attempt = ExamAttempt::where('exam_id', $exam->id)
        ->where('user_id', $user->id)
        ->firstOrFail();

      // Check if already completed
      if ($attempt->completed_at) {
        return response()->json([
          'success' => false,
          'message' => 'هذا الاختبار تم إرساله بالفعل',
        ], 400);
      }

      // Check if time has expired
      $isExpired = $this->examService->isExamTimeExpired($attempt);
      $autoSubmit = $request->input('auto_submit', false) || $isExpired;

      // If time expired but not auto-submit request, force auto-submit
      if ($isExpired && !$request->input('auto_submit', false)) {
        return response()->json([
          'success' => false,
          'message' => 'انتهى وقت الاختبار',
          'time_expired' => true,
        ], 400);
      }

      $answers = $request->input('answers', []);
      $attempt = $this->examService->completeExamAttempt($attempt, $answers, $autoSubmit);

      // Check if results should be auto-published
      $autoPublishResults = \App\Models\Setting::get('auto_publish_results', false);
      $passingScore = \App\Models\Setting::get('passing_score', 70);

      $responseData = [
        'completed_at' => $attempt->completed_at,
        'auto_submitted' => $attempt->auto_submitted,
        'results_published' => $autoPublishResults,
      ];

      // Only include score if auto-publish is enabled
      if ($autoPublishResults) {
        $responseData['score'] = $attempt->score;
        $responseData['passing_score'] = $passingScore;
        $responseData['passed'] = $attempt->score >= $passingScore;
        $responseData['total_questions'] = count($attempt->exam->form_schema['questions'] ?? []);
      }

      return response()->json([
        'success' => true,
        'message' => $autoSubmit ? 'تم إرسال الاختبار تلقائياً' : 'تم إرسال الاختبار بنجاح',
        'data' => $responseData,
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Admin: Create a new exam
   */
  public function store(Request $request): JsonResponse
  {
    $request->validate([
      'title' => 'required|string|max:255',
      'description' => 'nullable|string',
      'form_schema' => 'required|array',
      'deadline_at' => 'nullable|date',
    ]);

    $admin = Auth::user();
    $exam = Exam::create([
      'title' => $request->title,
      'description' => $request->description,
      'form_schema' => $request->form_schema,
      'deadline_at' => $request->deadline_at,
      'max_attempts' => 1,
    ]);

    $this->auditService->log(
      causer: $admin,
      action: 'exam.created',
      subject: $exam,
      meta: ['title' => $exam->title]
    );

    return response()->json([
      'success' => true,
      'message' => 'Exam created successfully',
      'data' => $exam,
    ], 201);
  }

  /**
   * Admin: Get all exams
   */
  public function adminIndex(): JsonResponse
  {
    $exams = Exam::withCount('attempts')->orderBy('created_at', 'desc')->get();

    return response()->json([
      'success' => true,
      'data' => $exams,
    ]);
  }

  /**
   * Admin: Get a specific exam with details
   */
  public function adminShow(Exam $exam): JsonResponse
  {
    $exam->loadCount('attempts');

    return response()->json([
      'success' => true,
      'data' => $exam,
    ]);
  }

  /**
   * Admin: Update an exam
   */
  public function update(Request $request, Exam $exam): JsonResponse
  {
    $request->validate([
      'title' => 'sometimes|string|max:255',
      'description' => 'nullable|string',
      'form_schema' => 'sometimes|array',
      'deadline_at' => 'nullable|date',
    ]);

    $admin = Auth::user();
    $exam->update($request->only(['title', 'description', 'form_schema', 'deadline_at']));

    $this->auditService->log(
      causer: $admin,
      action: 'exam.updated',
      subject: $exam,
      meta: ['title' => $exam->title]
    );

    return response()->json([
      'success' => true,
      'message' => 'Exam updated successfully',
      'data' => $exam,
    ]);
  }

  /**
   * Admin: Delete an exam
   */
  public function destroy(Exam $exam): JsonResponse
  {
    // Check if exam has attempts
    if ($exam->attempts()->count() > 0) {
      return response()->json([
        'success' => false,
        'message' => 'Cannot delete exam with existing attempts',
      ], 422);
    }

    $admin = Auth::user();
    $this->auditService->log(
      causer: $admin,
      action: 'exam.deleted',
      subject: $exam,
      meta: ['title' => $exam->title]
    );

    $exam->delete();

    return response()->json([
      'success' => true,
      'message' => 'Exam deleted successfully',
    ]);
  }

  /**
   * Admin: Publish an exam
   */
  public function publish(Exam $exam): JsonResponse
  {
    $admin = Auth::user();
    $exam = $this->examService->publishExam($exam, $admin);

    return response()->json([
      'success' => true,
      'message' => 'Exam published successfully',
      'data' => $exam,
    ]);
  }

  /**
   * Get detailed exam results with answers and correct answers
   */
  public function results(Exam $exam): JsonResponse
  {
    $user = Auth::user();
    $attempt = $this->examService->getUserExamAttempt($exam, $user);

    // Check if attempt exists and is completed
    if (!$attempt || !$attempt->completed_at) {
      return response()->json([
        'success' => false,
        'message' => 'لم يتم إكمال هذا الاختبار',
      ], 404);
    }

    // Check if results should be published
    $autoPublishResults = \App\Models\Setting::get('auto_publish_results', false);
    if (!$autoPublishResults) {
      return response()->json([
        'success' => false,
        'message' => 'لم يتم نشر النتائج بعد',
      ], 403);
    }

    $passingScore = \App\Models\Setting::get('passing_score', 70);
    $userAnswers = $attempt->answers ?? [];

    // Build detailed results with questions and answers
    $questions = $exam->form_schema['questions'] ?? [];
    $detailedQuestions = array_map(function ($question) use ($userAnswers) {
      $questionId = $question['id'];
      $userAnswer = $userAnswers[$questionId] ?? null;
      $correctAnswer = $question['correct_answer'] ?? null;

      // Determine if answer is correct
      $isCorrect = $userAnswer && $correctAnswer && $userAnswer === $correctAnswer;

      return [
        'id' => $question['id'],
        'text' => $question['text'] ?? $question['question'] ?? '',
        'type' => $question['type'] ?? 'multiple_choice',
        'points' => $question['points'] ?? 1,
        'options' => $question['options'] ?? [],
        'user_answer' => $userAnswer,
        'correct_answer' => $correctAnswer,
        'is_correct' => $isCorrect,
      ];
    }, $questions);

    return response()->json([
      'success' => true,
      'data' => [
        'exam' => [
          'id' => $exam->id,
          'title' => $exam->title,
          'description' => $exam->description,
        ],
        'attempt' => [
          'id' => $attempt->id,
          'score' => $attempt->score,
          'completed_at' => $attempt->completed_at,
          'auto_submitted' => $attempt->auto_submitted,
        ],
        'results' => [
          'score' => $attempt->score,
          'passing_score' => $passingScore,
          'passed' => $attempt->score >= $passingScore,
          'total_questions' => count($detailedQuestions),
          'correct_answers' => collect($detailedQuestions)->filter(fn($q) => $q['is_correct'])->count(),
        ],
        'questions' => $detailedQuestions,
      ],
    ]);
  }

  /**
   * Admin: Unpublish an exam
   */
  public function unpublish(Exam $exam): JsonResponse
  {
    $admin = Auth::user();
    $exam->update([
      'is_published' => false,
    ]);

    $this->auditService->log(
      causer: $admin,
      action: 'exam.unpublished',
      subject: $exam,
      meta: ['title' => $exam->title]
    );

    return response()->json([
      'success' => true,
      'message' => 'Exam unpublished successfully',
      'data' => $exam,
    ]);
  }

  /**
   * Admin: Get all attempts for an exam
   */
  public function attempts(Exam $exam): JsonResponse
  {
    $attempts = ExamAttempt::where('exam_id', $exam->id)
      ->with('user:id,email,full_name')
      ->get();

    return response()->json([
      'success' => true,
      'data' => $attempts,
    ]);
  }
}
