<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\EvaluationSheet;
use App\Services\EvaluationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class EvaluationController extends Controller
{
  public function __construct(
    protected EvaluationService $evaluationService,
    protected \App\Services\AuditService $auditService,
  ) {}

  /**
   * Judge: Get all active evaluation sheets (for selection)
   */
  public function sheets(): JsonResponse
  {
    $sheets = EvaluationSheet::where('is_active', true)
      ->with(['createdBy:id,full_name'])
      ->withCount('evaluations')
      ->get();

    return response()->json([
      'success' => true,
      'data' => $sheets,
    ]);
  }

  /**
   * Admin: Create an evaluation sheet
   */
  public function createSheet(Request $request): JsonResponse
  {
    $request->validate([
      'title' => 'required|string|max:255',
      'criteria' => 'required|array',
      'criteria.*.name' => 'required|string',
      'criteria.*.max_score' => 'required|numeric|min:0',
      'is_active' => 'boolean',
    ]);

    $admin = Auth::user();

    $sheet = EvaluationSheet::create([
      'created_by' => $admin->id,
      'title' => $request->title,
      'criteria' => $request->criteria,
      'is_active' => $request->boolean('is_active', true),
    ]);

    $this->auditService->log(
      causer: $admin,
      action: 'evaluation_sheet.created',
      subject: $sheet,
      meta: ['title' => $sheet->title, 'criteria_count' => count($sheet->criteria)]
    );

    return response()->json([
      'success' => true,
      'message' => 'Evaluation sheet created successfully',
      'data' => $sheet,
    ], 201);
  }

  /**
   * Admin: Update an evaluation sheet
   */
  public function updateSheet(Request $request, EvaluationSheet $sheet): JsonResponse
  {
    $request->validate([
      'title' => 'sometimes|required|string|max:255',
      'criteria' => 'sometimes|required|array',
      'criteria.*.name' => 'required|string',
      'criteria.*.max_score' => 'required|numeric|min:0',
      'is_active' => 'boolean',
    ]);

    $admin = Auth::user();

    $updateData = $request->only(['title', 'criteria', 'is_active']);
    $sheet->update($updateData);

    $this->auditService->log(
      causer: $admin,
      action: 'evaluation_sheet.updated',
      subject: $sheet,
      meta: ['title' => $sheet->title]
    );

    return response()->json([
      'success' => true,
      'message' => 'Evaluation sheet updated successfully',
      'data' => $sheet,
    ]);
  }

  /**
   * Admin: Delete an evaluation sheet
   */
  public function deleteSheet(EvaluationSheet $sheet): JsonResponse
  {
    $admin = Auth::user();

    // Check if sheet has evaluations
    if ($sheet->evaluations()->count() > 0) {
      return response()->json([
        'success' => false,
        'message' => 'Cannot delete evaluation sheet that has associated evaluations',
      ], 409);
    }

    $this->auditService->log(
      causer: $admin,
      action: 'evaluation_sheet.deleted',
      subject: $sheet,
      meta: ['title' => $sheet->title]
    );

    $sheet->delete();

    return response()->json([
      'success' => true,
      'message' => 'Evaluation sheet deleted successfully',
    ]);
  }

  /**
   * Admin: Get all evaluation sheets (including inactive ones)
   */
  public function allSheets(): JsonResponse
  {
    $sheets = EvaluationSheet::with(['createdBy:id,full_name'])
      ->withCount('evaluations')
      ->latest()
      ->get();

    return response()->json([
      'success' => true,
      'data' => $sheets,
    ]);
  }

  /**
   * Judge: Get users available for evaluation
   */
  public function availableUsers(Request $request): JsonResponse
  {
    $perPage = $request->query('per_page', 15);
    $page = $request->query('page', 1);
    $search = $request->query('search');

    $query = User::where('role', 'user')
      ->where(function ($q) {
        $q->where('status', 'exam_completed')
          ->orWhere('status', 'under_evaluation')
          ->orWhere('status', 'evaluated');
      })
      ->with(['examAttempts' => function ($q) {
        $q->whereNotNull('completed_at')->latest();
      }, 'applicant.documents'])->withCount(['evaluations' => function ($q) {
        $q->whereNotNull('submitted_at');
      }]);

    // Handle search
    if ($search) {
      $query->where(function ($q) use ($search) {
        $q->where('full_name', 'like', "%{$search}%")
          ->orWhere('email', 'like', "%{$search}%");
      });
    }

    // Paginate
    $users = $query->paginate($perPage, ['*'], 'page', $page);

    return response()->json([
      'success' => true,
      'data' => $users->items(),
      'pagination' => [
        'total' => $users->total(),
        'count' => $users->count(),
        'per_page' => $users->perPage(),
        'current_page' => $users->currentPage(),
        'last_page' => $users->lastPage(),
        'from' => $users->firstItem(),
        'to' => $users->lastItem(),
      ],
    ], 200);
  }

  /**
   * Judge: Submit an evaluation for a user
   */
  public function submit(Request $request): JsonResponse
  {
    $request->validate([
      'evaluation_sheet_id' => 'required|exists:evaluation_sheets,id',
      'evaluated_user_id' => 'required|exists:users,id',
      'answers' => 'required|array',
    ]);

    try {
      $judge = Auth::user();
      $evaluatedUser = User::findOrFail($request->evaluated_user_id);

      $evaluation = $this->evaluationService->submitEvaluation(
        $request->all(),
        $judge,
        $evaluatedUser
      );

      return response()->json([
        'success' => true,
        'message' => 'Evaluation submitted successfully',
        'data' => $evaluation,
      ], 201);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Judge: Evaluate a specific user (with criteria scores)
   */
  public function evaluateUser(Request $request, User $user): JsonResponse
  {
    $request->validate([
      'evaluation_sheet_id' => 'required|exists:evaluation_sheets,id',
      'scores' => 'required|array',
      'notes' => 'nullable|string',
    ]);

    try {
      $judge = Auth::user();

      // Get the evaluation sheet to map scores to criteria
      $sheet = EvaluationSheet::findOrFail($request->evaluation_sheet_id);

      // Convert scores to answers format (if needed by the service)
      $answers = [];
      foreach ($request->scores as $criterionName => $score) {
        $answers[$criterionName] = $score;
      }

      $evaluation = $this->evaluationService->submitEvaluation(
        [
          'evaluation_sheet_id' => $request->evaluation_sheet_id,
          'evaluated_user_id' => $user->id,
          'answers' => $answers,
          'notes' => $request->notes,
        ],
        $judge,
        $user
      );

      return response()->json([
        'success' => true,
        'message' => 'Evaluation submitted successfully',
        'data' => $evaluation,
      ], 201);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Get all evaluations for a user
   */
  public function userEvaluations(User $user): JsonResponse
  {
    $evaluations = $this->evaluationService->getUserEvaluations($user);

    return response()->json([
      'success' => true,
      'data' => [
        'count' => $evaluations->count(),
        'max_allowed' => 5,
        'evaluations' => $evaluations,
      ],
    ]);
  }

  /**
   * Judge: Get my evaluations (evaluations submitted by current judge)
   */
  public function myEvaluations(): JsonResponse
  {
    $judge = Auth::user();

    $evaluations = \App\Models\Evaluation::where('judge_id', $judge->id)
      ->whereNotNull('submitted_at')
      ->with(['evaluatedUser', 'evaluationSheet'])
      ->latest('submitted_at')
      ->get()
      ->map(function ($evaluation) {
        // Calculate max score from criteria array
        $criteria = $evaluation->evaluationSheet->criteria ?? [];
        $maxScore = 0;
        if (is_array($criteria)) {
          foreach ($criteria as $criterion) {
            $maxScore += $criterion['max_score'] ?? 0;
          }
        }

        return [
          'id' => $evaluation->id,
          'user_id' => $evaluation->evaluated_user_id,
          'user_name' => $evaluation->evaluatedUser->full_name,
          'user_email' => $evaluation->evaluatedUser->email,
          'evaluation_sheet_title' => $evaluation->evaluationSheet->title,
          'score' => $evaluation->score,
          'max_score' => $maxScore,
          'submitted_at' => $evaluation->submitted_at,
        ];
      });

    return response()->json([
      'success' => true,
      'data' => $evaluations,
    ]);
  }

  /**
   * Judge: Get evaluation statistics
   */
  public function statistics(): JsonResponse
  {
    $judge = Auth::user();

    /**
     * @var User $judge
     */

    // Get total assigned users count
    $totalAssignedUsers = \App\Models\User::where('role', 'user')
      ->whereHas('evaluations', function ($query) use ($judge) {
        $query->where('judge_id', $judge->id);
      })
      ->count();

    // Get evaluations with submitted status
    $completedEvaluations = $judge->givenEvaluations()
      ->whereNotNull('submitted_at')
      ->count();

    // Get pending evaluations (not yet submitted)
    $pendingEvaluations = $judge->givenEvaluations()
      ->whereNull('submitted_at')
      ->count();

    // Calculate average score
    $evaluationsWithScores = $judge->givenEvaluations()
      ->whereNotNull('submitted_at')
      ->whereNotNull('score')
      ->get();

    $avgScore = $evaluationsWithScores->count() > 0
      ? $evaluationsWithScores->avg('score')
      : 0;

    $stats = [
      'total_assigned_users' => $totalAssignedUsers,
      'pending_evaluations' => $pendingEvaluations,
      'completed_evaluations' => $completedEvaluations,
      'avg_score_given' => round($avgScore, 2),
      'evaluation_sheets_count' => $judge->createdEvaluationSheets()->count(),
      'total_evaluations' => $judge->givenEvaluations()->count(),
      'total_sheets' => $judge->createdEvaluationSheets()->count(),
      'evaluations_this_month' => $judge->givenEvaluations()
        ->whereMonth('submitted_at', now()->month)
        ->whereNotNull('submitted_at')
        ->count(),
    ];

    return response()->json([
      'success' => true,
      'data' => $stats,
    ]);
  }
}
