<?php

namespace App\Http\Controllers;

use App\Models\ApplicantDocument;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DocumentController extends Controller
{
  public function __construct(
    protected \App\Services\AuditService $auditService,
  ) {}
  /**
   * Download a document with authorization check
   */
  public function download(ApplicantDocument $document): StreamedResponse
  {
    // Get the document's applicant
    $applicant = $document->applicant;

    // Check if authenticated user owns this document
    $user = Auth::user();
    if ($applicant->user_id !== $user->id && $user->role !== 'admin') {
      abort(403, 'Unauthorized to access this document');
    }

    // Check if file exists
    if (!Storage::exists($document->file_path)) {
      abort(404, 'Document file not found');
    }

    // Log document download
    $this->auditService->log(
      causer: $user,
      action: 'document.downloaded',
      subject: $document,
      meta: [
        'document_id' => $document->id,
        'type' => $document->type,
        'applicant_id' => $applicant->id,
      ]
    );

    // Get file content
    $fileContent = Storage::get($document->file_path);
    $fileName = basename($document->file_path);
    $mimeType = Storage::mimeType($document->file_path);

    return response()->streamDownload(
      function () use ($fileContent) {
        echo $fileContent;
      },
      $fileName,
      [
        'Content-Type' => $mimeType,
        'Content-Disposition' => "inline; filename=\"{$fileName}\"",
      ]
    );
  }

  /**
   * Preview a document (returns file content with appropriate headers)
   */
  public function preview(ApplicantDocument $document)
  {
    // Get the document's applicant
    $applicant = $document->applicant;

    // Check if authenticated user owns this document
    $user = Auth::user();
    if (!$user || ($applicant->user_id !== $user->id && $user->role !== 'admin')) {
      abort(403, 'Unauthorized to access this document');
    }

    // Check if file exists
    if (!Storage::exists($document->file_path)) {
      abort(404, 'Document file not found');
    }

    return Storage::response($document->file_path, null, [
      'Content-Disposition' => 'inline'
    ]);
  }

  /**
   * Get document metadata
   */
  public function show(ApplicantDocument $document)
  {
    // Get the document's applicant
    $applicant = $document->applicant;

    // Check if authenticated user owns this document
    $user = Auth::user();
    if ($applicant->user_id !== $user->id && $user->role !== 'admin') {
      abort(403, 'Unauthorized to access this document');
    }

    return response()->json([
      'success' => true,
      'data' => [
        'id' => $document->id,
        'type' => $document->type,
        'file_path' => $document->file_path,
        'file_size' => Storage::exists($document->file_path)
          ? Storage::size($document->file_path)
          : 0,
        'mime_type' => Storage::mimeType($document->file_path),
        'created_at' => $document->created_at,
        'applicant_id' => $applicant->id,
      ],
    ]);
  }

  /**
   * Judge: Download a document for a user being evaluated
   */
  public function judgeDownload($userId, ApplicantDocument $document)
  {
    // Get the document's applicant (the user being evaluated)
    $applicant = $document->applicant;
    $user = Auth::user();

    // Check if userId matches the document's applicant user
    if ((int)$userId !== $applicant->user_id) {
      abort(404, 'Document not found');
    }

    // Check if judge is evaluating this user (check in evaluations table)
    $isEvaluatingUser = \App\Models\Evaluation::where('judge_id', $user->id)
      ->where('evaluated_user_id', $applicant->user_id)
      ->exists();

    if (!$isEvaluatingUser) {
      abort(403, 'Unauthorized to access this document');
    }

    // Check if file exists
    if (!Storage::exists($document->file_path)) {
      abort(404, 'Document file not found');
    }

    // Log document download
    $this->auditService->log(
      causer: $user,
      action: 'document.downloaded',
      subject: $document,
      meta: [
        'document_id' => $document->id,
        'type' => $document->type,
        'applicant_id' => $applicant->id,
        'accessed_by_judge' => true,
      ]
    );

    // Get file content
    $fileContent = Storage::get($document->file_path);
    $fileName = basename($document->file_path);
    $mimeType = Storage::mimeType($document->file_path);

    return response()->streamDownload(
      function () use ($fileContent) {
        echo $fileContent;
      },
      $fileName,
      [
        'Content-Type' => $mimeType,
        'Content-Disposition' => "inline; filename=\"{$fileName}\"",
      ]
    );
  }

  /**
   * Judge: Preview a document for a user being evaluated
   */
  public function judgePreview($userId, ApplicantDocument $document)
  {
    // Get the document's applicant (the user being evaluated)
    $applicant = $document->applicant;
    $user = Auth::user();

    // Check if userId matches the document's applicant user
    if ((int)$userId !== $applicant->user_id) {
      abort(404, 'Document not found');
    }

    // Check if judge is evaluating this user (check in evaluations table)
    $isEvaluatingUser = \App\Models\Evaluation::where('judge_id', $user->id)
      ->where('evaluated_user_id', $applicant->user_id)
      ->exists();

    if (!$isEvaluatingUser) {
      abort(403, 'Unauthorized to access this document');
    }

    // Check if file exists
    if (!Storage::exists($document->file_path)) {
      abort(404, 'Document file not found');
    }

    return Storage::response($document->file_path, null, [
      'Content-Disposition' => 'inline'
    ]);
  }

  /**
   * Judge: Get document metadata for a user being evaluated
   */
  public function judgeShow($userId, ApplicantDocument $document)
  {
    // Get the document's applicant (the user being evaluated)
    $applicant = $document->applicant;
    $user = Auth::user();

    // Check if userId matches the document's applicant user
    if ((int)$userId !== $applicant->user_id) {
      abort(404, 'Document not found');
    }

    // Check if judge is evaluating this user (check in evaluations table)
    $isEvaluatingUser = \App\Models\Evaluation::where('judge_id', $user->id)
      ->where('evaluated_user_id', $applicant->user_id)
      ->exists();

    if (!$isEvaluatingUser) {
      abort(403, 'Unauthorized to access this document');
    }

    return response()->json([
      'success' => true,
      'data' => [
        'id' => $document->id,
        'type' => $document->type,
        'file_path' => $document->file_path,
        'file_size' => Storage::exists($document->file_path)
          ? Storage::size($document->file_path)
          : 0,
        'mime_type' => Storage::mimeType($document->file_path),
        'created_at' => $document->created_at,
        'applicant_id' => $applicant->id,
      ],
    ]);
  }

  /**
   * Get document by type for preview (public preview for judge)
   */
  public function previewByType($userId, $documentType)
  {
    $authUser = Auth::user();

    // Get user and applicant
    $targetUser = \App\Models\User::find($userId);
    if (!$targetUser || !$targetUser->applicant) {
      abort(404, 'User or applicant not found');
    }

    $applicant = $targetUser->applicant;

    // Check authorization - must be authenticated
    if (!$authUser) {
      abort(401, 'Unauthorized');
    }

    // Only judges and admins can preview documents
    if (!in_array($authUser->role, ['admin', 'judge'])) {
      abort(403, 'Forbidden');
    }

    // Find document by type
    $document = $applicant->documents()
      ->where('type', $documentType)
      ->first();

    if (!$document || !Storage::exists($document->file_path)) {
      abort(404, 'Document not found');
    }

    return Storage::response($document->file_path, null, [
      'Content-Disposition' => 'inline'
    ]);
  }

  /**
   * Admin: Get document by type for preview
   */
  public function adminPreviewByType($userId, $documentType)
  {
    $authUser = Auth::user();

    // Check authorization - must be admin
    if (!$authUser || $authUser->role !== 'admin') {
      abort(403, 'Forbidden - Admin access required');
    }

    // Get user and applicant
    $targetUser = \App\Models\User::find($userId);
    if (!$targetUser || !$targetUser->applicant) {
      abort(404, 'User or applicant not found');
    }

    $applicant = $targetUser->applicant;

    // Find document by type
    $document = $applicant->documents()
      ->where('type', $documentType)
      ->first();

    if (!$document || !Storage::exists($document->file_path)) {
      abort(404, 'Document not found');
    }

    return Storage::response($document->file_path, null, [
      'Content-Disposition' => 'inline'
    ]);
  }
}
