<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\AuditService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
  public function __construct(
    protected AuditService $auditService,
  ) {}

  /**
   * Login user and return token
   */
  public function login(Request $request): JsonResponse
  {
    $request->validate([
      'email' => 'required|email',
      'password' => 'required',
    ]);

    $email = strtolower(trim($request->email));
    $ip = $request->ip();

    // Rate limiting keys
    $ipKey = 'login-ip:' . $ip;
    $emailKey = 'login-email:' . $email;

    // Check IP-based rate limiting (prevents brute force attacks)
    if (RateLimiter::tooManyAttempts($ipKey, 10)) {
      $seconds = RateLimiter::availableIn($ipKey);
      $minutes = ceil($seconds / 60);

      return response()->json([
        'success' => false,
        'message' => "تم تجاوز الحد الأقصى من المحاولات من عنوان IP هذا. حاول مرة أخرى بعد {$minutes} دقيقة.",
        'retry_after' => $seconds,
      ], 429);
    }

    $user = User::where('email', $email)->first();

    // Check if user exists
    if (!$user) {
      // Hit rate limiters even for non-existent users to prevent enumeration
      RateLimiter::hit($ipKey, 900); // 15 minutes
      RateLimiter::hit($emailKey, 900);

      throw ValidationException::withMessages([
        'email' => ['بيانات الدخول غير صحيحة.'],
      ]);
    }

    // Check if account is locked due to too many failed attempts
    if ($user->locked_until && now()->lt($user->locked_until)) {
      $remainingMinutes = now()->diffInMinutes($user->locked_until, true) + 1;

      $this->auditService->log(
        causer: null,
        action: 'user.login_attempt_while_locked',
        subject: $user,
        meta: [
          'ip' => $ip,
          'locked_until' => $user->locked_until,
        ]
      );

      return response()->json([
        'success' => false,
        'message' => "حسابك مقفل مؤقتاً بسبب المحاولات الفاشلة المتعددة. حاول مرة أخرى بعد {$remainingMinutes} دقيقة.",
        'locked_until' => $user->locked_until,
      ], 423);
    }

    // Reset lock if time has expired
    if ($user->locked_until && now()->gte($user->locked_until)) {
      $user->update([
        'login_attempts' => 0,
        'locked_until' => null,
      ]);
    }

    // Verify password
    if (!Hash::check($request->password, $user->password)) {
      // Hit rate limiters
      RateLimiter::hit($ipKey, 900); // 15 minutes
      RateLimiter::hit($emailKey, 1800); // 30 minutes

      // Increment login attempts atomically
      $user->increment('login_attempts');
      $user->refresh(); // Refresh to get the updated value

      $maxAttempts = \App\Models\Setting::get('max_login_attempts', 5);
      $remainingAttempts = max(0, $maxAttempts - $user->login_attempts);

      // Lock account if max attempts reached
      if ($user->login_attempts >= $maxAttempts) {
        $lockDuration = 30; // minutes
        $user->update([
          'locked_until' => now()->addMinutes($lockDuration),
        ]);

        $this->auditService->log(
          causer: null,
          action: 'user.account_locked',
          subject: $user,
          meta: [
            'reason' => 'max_login_attempts',
            'attempts' => $user->login_attempts,
            'ip' => $ip,
            'lock_duration_minutes' => $lockDuration,
          ]
        );

        return response()->json([
          'success' => false,
          'message' => "تم قفل حسابك بسبب المحاولات الفاشلة المتعددة. حاول مرة أخرى بعد {$lockDuration} دقيقة.",
          'locked_until' => $user->locked_until,
        ], 423);
      }

      // Log failed attempt
      $this->auditService->log(
        causer: null,
        action: 'user.login_failed',
        subject: $user,
        meta: [
          'ip' => $ip,
          'attempts' => $user->login_attempts,
          'remaining_attempts' => $remainingAttempts,
        ]
      );

      throw ValidationException::withMessages([
        'email' => ["بيانات الدخول غير صحيحة. المحاولات المتبقية: {$remainingAttempts}"],
      ]);
    }

    // Check if account is active
    if (!$user->is_active) {
      $this->auditService->log(
        causer: null,
        action: 'user.login_attempt_inactive',
        subject: $user,
        meta: ['ip' => $ip]
      );

      return response()->json([
        'success' => false,
        'message' => 'حسابك غير نشط. يرجى التواصل مع الدعم الفني.',
      ], 403);
    }

    // Successful login - Reset everything
    $user->update([
      'login_attempts' => 0,
      'locked_until' => null,
      'last_login_at' => now(),
      'last_login_ip' => $ip,
    ]);

    // Clear rate limiters for this user
    RateLimiter::clear($emailKey);

    // Create token
    $token = $user->createToken('api-token', ['*'], now()->addDays(30))->plainTextToken;

    // Log successful login
    $this->auditService->log(
      causer: $user,
      action: 'user.login',
      subject: $user,
      meta: ['ip' => $ip]
    );

    return response()->json([
      'success' => true,
      'message' => 'تم تسجيل الدخول بنجاح',
      'data' => [
        'user' => [
          'id' => $user->id,
          'email' => $user->email,
          'full_name' => $user->full_name,
          'role' => $user->role,
          'status' => $user->status,
        ],
        'token' => $token,
      ],
    ]);
  }

  /**
   * Logout user (revoke token)
   */
  public function logout(Request $request): JsonResponse
  {
    $user = $request->user();

    // Revoke current token
    $request->user()->currentAccessToken()->delete();

    $this->auditService->log(
      causer: $user,
      action: 'user.logout',
      subject: $user,
      meta: ['ip' => $request->ip()]
    );

    return response()->json([
      'success' => true,
      'message' => 'تم تسجيل الخروج بنجاح',
    ]);
  }

  /**
   * Get authenticated user info
   */
  public function me(Request $request): JsonResponse
  {
    $user = $request->user();

    return response()->json([
      'success' => true,
      'data' => [
        'id' => $user->id,
        'email' => $user->email,
        'full_name' => $user->full_name,
        'role' => $user->role,
        'status' => $user->status,
        'is_active' => $user->is_active,
        'last_login_at' => $user->last_login_at,
        'categories' => $user->categories->pluck('name'),
        'has_completed_exam' => $user->hasCompletedExam(),
        'evaluation_count' => $user->getEvaluationCount(),
        'is_evaluated' => $user->isEvaluated(),
      ],
    ]);
  }

  /**
   * Change password
   */
  public function changePassword(Request $request): JsonResponse
  {
    $request->validate([
      'current_password' => 'required',
      'new_password' => 'required|min:8|confirmed',
    ]);

    $user = $request->user();

    if (!Hash::check($request->current_password, $user->password)) {
      return response()->json([
        'success' => false,
        'message' => 'كلمة المرور الحالية غير صحيحة',
      ], 400);
    }

    $user->update([
      'password' => Hash::make($request->new_password),
    ]);

    // Revoke all other tokens except current one
    $user->tokens()->where('id', '!=', $request->user()->currentAccessToken()->id)->delete();

    $this->auditService->log(
      causer: $user,
      action: 'user.password_changed',
      subject: $user,
      meta: ['ip' => $request->ip()]
    );

    return response()->json([
      'success' => true,
      'message' => 'تم تغيير كلمة المرور بنجاح',
    ]);
  }

  /**
   * Request password reset
   */
  public function forgotPassword(Request $request): JsonResponse
  {
    $request->validate([
      'email' => 'required|email',
    ]);

    // Rate limit password reset requests
    $key = 'password-reset:' . $request->ip();

    if (RateLimiter::tooManyAttempts($key, 3)) {
      $seconds = RateLimiter::availableIn($key);
      return response()->json([
        'success' => false,
        'message' => 'تم تجاوز الحد الأقصى من محاولات إعادة تعيين كلمة المرور. حاول مرة أخرى لاحقاً.',
      ], 429);
    }

    RateLimiter::hit($key, 3600); // 1 hour

    // TODO: Implement password reset email

    return response()->json([
      'success' => true,
      'message' => 'إذا كان البريد الإلكتروني مسجلاً، ستصلك تعليمات إعادة تعيين كلمة المرور.',
    ]);
  }
}
