<?php

namespace App\Http\Controllers;

use App\Models\Applicant;
use App\Models\User;
use App\Models\ApplicantSubmission;
use App\Events\UserRegistered;
use App\Services\AuditService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Http\Requests\StoreApplicantRequest;

class ApplicantController extends Controller
{
    public function __construct(
        protected AuditService $auditService,
    ) {}

    /**
     * List applicants with relations
     */
    public function index()
    {
        return Applicant::with([
            'nationalities',
            'languages',
            'degrees',
            'documents',
        ])->paginate(20);
    }

    /**
     * Store new applicant and create user account
     */
    public function store(StoreApplicantRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            // Check if applicant already exists with this email or national_id
            $existingApplicant = Applicant::withTrashed()
                ->where('email', $data['email'])
                ->orWhere('national_id', $data['national_id'])
                ->first();

            if ($existingApplicant) {
                return response()->json([
                    'success' => false,
                    'message' => 'أنت مسجل بالفعل في هذا البرنامج. لا يمكن التقديم مرة أخرى.',
                    'error_code' => 'ALREADY_APPLIED',
                    'applicant_code' => $existingApplicant->applicant_code,
                ], 409);
            }

            // Generate applicant code
            $data['applicant_code'] = 'APP-' . strtoupper(Str::random(6));

            // Create applicant
            $applicant = Applicant::create($data);

            /*
            |--------------------------------------------------------------------------
            | Create User Account
            |--------------------------------------------------------------------------
            */
            $mobile_last4 = substr($data['mobile'], -4);
            $national_last4 = substr($data['national_id'], -4);

            $hashed_value = Hash::make($mobile_last4 . '_EYDP#' . $national_last4);
            $user = User::firstOrCreate(
                ['email' => $data['email']],
                [
                    'email' => $data['email'],
                    'national_id' => encrypt($data['national_id']),
                    'full_name' => $data['full_name_en'],
                    'role' => 'user',
                    'is_active' => false,
                    'status' => 'pending',
                    'password' => $hashed_value,
                ]
            );

            // Link applicant to user
            $applicant->update(['user_id' => $user->id]);

            /*
            |--------------------------------------------------------------------------
            | Save Nationalities
            |--------------------------------------------------------------------------
            */
            if (!empty($data['nationalities'])) {
                foreach ($data['nationalities'] as $nat) {
                    $applicant->nationalities()->create($nat);
                }
            }

            /*
        |--------------------------------------------------------------------------
        | Save Languages
        |--------------------------------------------------------------------------
        */
            if (!empty($data['languages'])) {
                foreach ($data['languages'] as $lang) {
                    $applicant->languages()->create($lang);
                }
            }

            /*
        |--------------------------------------------------------------------------
        | Save Degrees
        |--------------------------------------------------------------------------
        */
            if (!empty($data['degrees'])) {
                foreach ($data['degrees'] as $deg) {
                    $applicant->degrees()->create($deg);
                }
            }

            /*
        |--------------------------------------------------------------------------
        | Save Documents
        |--------------------------------------------------------------------------
        |
        | documents = [
        |   'national_id' => file,
        |   'cv' => file,
        |   'photo_casual' => [file, file, ...],
        |   'photo_official' => file,
        | ]
        |--------------------------------------------------------------------------
        */
            if (!empty($request->documents)) {
                foreach ($request->documents as $type => $files) {

                    // If multiple files (photo_casual)
                    if (is_array($files)) {
                        foreach ($files as $file) {
                            $path = $file->store("applicants/{$applicant->id}");
                            $applicant->documents()->create([
                                'type' => $type,
                                'file_path' => $path,
                            ]);
                        }

                        // Single file (national_id, cv, official_photo)
                    } else {
                        $path = $files->store("applicants/{$applicant->id}");
                        $applicant->documents()->create([
                            'type' => $type,
                            'file_path' => $path,
                        ]);
                    }
                }
            }

            /*
        |--------------------------------------------------------------------------
        | Load relations and return
        |--------------------------------------------------------------------------
        */
            $applicant->load(['nationalities.country', 'languages.language', 'degrees', 'documents', 'governorate']);

            /*
        |--------------------------------------------------------------------------
        | Track Submission by IP and Browser Fingerprint
        |--------------------------------------------------------------------------
        */
            $browserFingerprint = $request->input('browser_fingerprint');
            $deviceType = $request->input('device_type');
            $os = $request->input('os');
            $browser = $request->input('browser');

            if ($browserFingerprint) {
                ApplicantSubmission::create([
                    'applicant_id' => $applicant->id,
                    'ip_address' => $request->ip(),
                    'browser_fingerprint' => $browserFingerprint,
                    'user_agent' => $request->userAgent(),
                    'device_type' => $deviceType,
                    'os' => $os,
                    'browser' => $browser,
                ]);
            }

            /*
        |--------------------------------------------------------------------------
        | Trigger UserRegistered Event
        |--------------------------------------------------------------------------
        */
            event(new UserRegistered($user, null));

            /*
        |--------------------------------------------------------------------------
        | Log the registration
        |--------------------------------------------------------------------------
        */
            $this->auditService->log(
                causer: $user,
                action: 'applicant.registered',
                subject: $applicant,
                meta: [
                    'applicant_code' => $applicant->applicant_code,
                    'email' => $applicant->email,
                ]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Registration completed successfully. Login credentials have been sent to your email.',
                'data' => [
                    'applicant' => $applicant,
                    'user' => [
                        'id' => $user->id,
                        'email' => $user->email,
                        'full_name' => $user->full_name,
                    ],
                ],
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Registration failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Show single applicant
     */
    public function show(Applicant $applicant)
    {
        return $applicant->load(['nationalities', 'languages', 'degrees', 'documents']);
    }

    /**
     * Update applicant (simple update)
     */
    // public function update(UpdateApplicantRequest $request, Applicant $applicant)
    // {
    //     $applicant->update($request->validated());

    //     return $this->show($applicant);
    // }

    /**
 * Delete applicant
 */
    // public function destroy(Applicant $applicant)
    // {
    //     $applicant->delete();
    //     return response()->json(['message' => 'deleted']);
    // }
}
