<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Applicant;
use App\Models\ApplicantSubmission;
use Illuminate\Http\Request;

class RegistrationCheckController extends Controller
{
  /**
   * Check if IP/Browser Fingerprint has already submitted
   * This is a server-side check that persists even if user clears localStorage
   */
  public function checkByDeviceFingerprint(Request $request)
  {
    $request->validate([
      'browser_fingerprint' => ['required', 'string'],
    ]);

    $ipAddress = $request->ip();
    $browserFingerprint = $request->input('browser_fingerprint');

    // Check if this IP or fingerprint has already submitted
    $previousSubmission = ApplicantSubmission::where(function ($query) use ($ipAddress, $browserFingerprint) {
      $query->where('ip_address', $ipAddress)
        ->orWhere('browser_fingerprint', $browserFingerprint);
    })->first();

    if ($previousSubmission && $previousSubmission->applicant) {
      return response()->json([
        'has_submitted' => true,
        'message' => 'تم اكتشاف أنك قدمت طلباً من قبل على هذا الجهاز.',
        'applicant_code' => $previousSubmission->applicant->applicant_code,
        'submitted_at' => $previousSubmission->created_at,
      ]);
    }

    return response()->json([
      'has_submitted' => false,
      'message' => 'لم يتم تقديم طلب من قبل من هذا الجهاز.',
    ]);
  }

  /**
   * Check if user has already applied
   * Uses email or national_id to detect previous applications
   */
  public function checkAlreadyApplied(Request $request)
  {
    $request->validate([
      'email' => ['required_without:national_id', 'email'],
      'national_id' => ['required_without:email', 'digits:14'],
    ]);

    $email = $request->input('email');
    $national_id = $request->input('national_id');

    // Check if applicant exists with this email or national_id
    $applicant = Applicant::withTrashed()
      ->where(function ($query) use ($email, $national_id) {
        if ($email) {
          $query->where('email', $email);
        }
        if ($national_id) {
          $query->orWhere('national_id', $national_id);
        }
      })
      ->first();

    if ($applicant) {
      return response()->json([
        'already_applied' => true,
        'message' => 'أنت مسجل بالفعل في هذا البرنامج. لا يمكن التقديم مرة أخرى.',
        'applicant_code' => $applicant->applicant_code,
      ]);
    }

    return response()->json([
      'already_applied' => false,
      'message' => 'يمكنك المتابعة مع الاستمارة',
    ]);
  }
}
